package ru.yandex.chemodan.app.migrator.sharpei.cleanup;

import java.util.concurrent.Semaphore;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.migrator.migration.UserMigrationManager;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author yashunsky
 */
@AllArgsConstructor
public class SharpeiCleanupManager {
    private static final Logger logger = LoggerFactory.getLogger(SharpeiCleanupManager.class);

    private final SharpeiCleanupDao sharpeiCleanupDao;
    private final UserMigrationManager userMigrationManager;
    private final SharpeiCleanupControl control;

    public boolean isUserReadOnly(DataApiUserId uid) {
        return userMigrationManager.isUserReadOnlySafe(uid, control.getSharpeiSemaphore());
    }

    public boolean removeIfNoData(DataApiUserId uid, boolean withRo) {
        if (!control.cleanupEnable()) {
            logger.info("User {} not removed due to disabled cleanup", uid);
            return false;
        }

        if (withRo) {
            userMigrationManager.checkSarpeiCacheSettings();
        }

        Semaphore sharpeiSemaphore = control.getSharpeiSemaphore();

        Option<Integer> userShardO = userMigrationManager.getUserShardO(uid, sharpeiSemaphore);

        if (!userShardO.isPresent()) {
            logger.info("User {} not registered in sharpei", uid);
            return false;
        }

        int userShard = userShardO.get();

        if (userMigrationManager.isAnyUserDataOnShard(uid, userShard)) {
            logger.info("User {} not removed as has data on shard {}", uid, userShard);
            return false;
        }

        if (withRo) {
            userMigrationManager.setReadOnly(uid, sharpeiSemaphore);

            ThreadUtils.sleep(control.getRoPause());

            if (userMigrationManager.isAnyUserDataOnShard(uid, userShard)) {
                userMigrationManager.removeReadOnly(uid, sharpeiSemaphore);
                logger.info("User {} not removed as data was found on shard {} after sleep ", uid, userShard);
                return false;
            }
        }

        boolean userWasDeleted = control.getRateLimiter()
                .acquirePermitAndExecute(1, () -> sharpeiCleanupDao.delete(uid, withRo));

        if (userWasDeleted) {
            logger.info("User {} was removed", uid);
        } else if (withRo) {
            logger.info("User {} not removed, removing RO", uid);
            userMigrationManager.removeReadOnly(uid, sharpeiSemaphore);
        } else {
            logger.info("User {} not removed", uid);
        }

        return userWasDeleted;
    }
}
