package ru.yandex.chemodan.app.migrator.tasks;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.migrator.MigratorTaskQueueName;
import ru.yandex.chemodan.app.migrator.control.BazingaDatasyncUserMigrationManager;
import ru.yandex.chemodan.app.migrator.logging.MigrationEventsLogger;
import ru.yandex.chemodan.app.migrator.logging.MigrationStageStatus;
import ru.yandex.commune.bazinga.BazingaStopExecuteException;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleConstant;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author yashunsky
 */
public abstract class DatasyncMigrationTaskSupport<T extends MigrationParameters> extends OnetimeTaskSupport<T> {
    private static final MigrationEventsLogger logger = new MigrationEventsLogger();
    private static final Logger generalLogger = LoggerFactory.getLogger(DatasyncMigrationTaskSupport.class);

    protected final BazingaDatasyncUserMigrationManager bazingaDatasyncUserMigrationManager;
    private final String sharpeiId;

    public DatasyncMigrationTaskSupport(T parameters, String sharpeiId) {
        super(parameters);
        this.bazingaDatasyncUserMigrationManager = null;
        this.sharpeiId = sharpeiId;
    }

    public DatasyncMigrationTaskSupport(
            Class<T> parametersClass,
            BazingaDatasyncUserMigrationManager bazingaDatasyncUserMigrationManager,
            String sharpeiId)
    {
        super(parametersClass);
        this.bazingaDatasyncUserMigrationManager = bazingaDatasyncUserMigrationManager;
        this.sharpeiId = sharpeiId;
    }

    abstract MigrationStageStatus executeInner(T parameters, ExecutionContext context) throws Exception;

    @Override
    protected void execute(T parameters, ExecutionContext context) throws Exception {
        ListF<Integer> requiredShards = parameters.getRequiredShards();
        while (!bazingaDatasyncUserMigrationManager.isMigrationPossible(requiredShards)) {
            generalLogger.warn("migration not possible for shards {}, awaiting 1s before retry", requiredShards);
            ThreadUtils.sleep(Duration.standardSeconds(1));
            ThreadLocalTimeout.check();
        }

        logger.log(sharpeiId, parameters, MigrationStageStatus.STARTED);
        try {
            MigrationStageStatus status = executeInner(parameters, context);
            logger.log(sharpeiId, parameters, status);
            if (status == MigrationStageStatus.ABORTED) {
                throw new BazingaStopExecuteException("Migration aborted");
            }
        } catch (Throwable t) {
            ExceptionUtils.throwIfUnrecoverable(t);
            if (!(t instanceof BazingaStopExecuteException)) {
                logger.log(sharpeiId, parameters, MigrationStageStatus.FAILED);
            }
            throw t;
        }
    }

    @Override
    public TaskId id() {
        return new TaskId(super.id().getId() + "_" + sharpeiId);
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public TaskQueueName queueName() {
        return MigratorTaskQueueName.MIGRATION;
    }

    @Override
    public ReschedulePolicy reschedulePolicy() {
        return new RescheduleConstant(Duration.standardMinutes(100), 100);
    }
}
