OK_STYLE = {
    'stroke': 'black',
    // 'fill': 'white',
    'stroke-width': 1
};
WARN_STYLE = {
    'stroke': '#ffd63a',
    // 'fill': '#ffd63a',
    'stroke-width': 2.5
};
CRIT_STYLE = {
    'stroke': '#f86c49',
    // 'fill': '#f86c49',
    'stroke-width': 5
};


function getRandomInt(min, max) {
    return Math.floor(Math.random() * (max - min + 1)) + min;
}

function d3Elem(element) {
    if (typeof element == 'string') {
        return d3.select(element)
    }
    return element
}

function getFloatAttr(elem, attr) {
    return parseFloat(elem.attr(attr))
}

function getScreenCoords(x, y, ctm) {
    var xn = ctm.e + x * ctm.a + y * ctm.c;
    var yn = ctm.f + x * ctm.b + y * ctm.d;
    return { x: xn, y: yn };
}

function getX(elem) {
    return getScreenCoords(getFloatAttr(elem, 'x'), getFloatAttr(elem, 'y'), elem.node().getCTM()).x
}

function getY(elem) {
    return getScreenCoords(getFloatAttr(elem, 'x'), getFloatAttr(elem, 'y'), elem.node().getCTM()).y
}

function getWidth(elem) {
    return getFloatAttr(elem, 'width')
}

function getHeight(elem) {
    return getFloatAttr(elem, 'height')
}

function isInsideRect(x, y, x1, y1, x2, y2) {
    return x >= x1 && x <= x2 && y >= y1 && y <= y2
}

function distancePointToD3Rect(x, y, d3Rect) {
    var x1 = getX(d3Rect);
    var x2 = x1 + getWidth(d3Rect);
    var y1 = getY(d3Rect);
    var y2 = y1 + getHeight(d3Rect);

    return distancePointToRect(x, y, x1, y1, x2, y2)
}

function findNearestRect(x, y, selector) {
    var nearest = undefined;
    var nearest_dist = undefined;

    d3.selectAll(selector).each(function (d, i) {
        var el = d3.select(this);
        var dist = distancePointToD3Rect(x, y, el);

        if (nearest == undefined || distTupleLess(dist, nearest_dist)) {
            nearest = el;
            nearest_dist = dist
        }
    });
    // console.log('found nearest ', x, y, nearest);
    // console.log('dist = ', nearest_dist);
    return nearest
}

function distancePointToRect(x, y, x1, y1, x2, y2) {
    var x_dist = Math.min(
        Math.abs(x - x1),
        Math.abs(x - x2)
    );
    var y_dist = Math.min(
        Math.abs(y - y1),
        Math.abs(y - y2)
    );
    if (isInsideRect(x, y, x1, y1, x2, y2)) {
        return [0, Math.min(x_dist, y_dist)]
    } else if (x >= x1 && x <= x2) {
        return [1, y_dist]
    } else if (y >= y1 && y <= y2) {
        return [1, x_dist]
    } else {
        return [1, x_dist + y_dist]
    }
}

function distTupleLess(t1, t2) {
    if (t1[0] != t2[0]) {
        return t1[0] < t2[0]
    }
    return t1[1] < t2[1]
}

$(document).ready(function() {
    var clusterState = {};
    var componentRects = {};
    var componentConnections = {};

    var svg = d3.select("#map").select("svg");

    window.componentRects = componentRects;

    var component_dc_info = d3.select("body")
        .append("div")
        .attr("id", "component_dc_info")
        .style("position", "absolute")
        .style("font", "12px sans-serif")
        .style("border-width", "1px")
        .style("border-style", "solid")
        .style("border-color", "grey")
        .style("border-radius", "8px")
        .style("pointer-events", "none")
        .style("opacity", 0);

    var component_version_info = d3.select("body")
        .append("div")
        .attr("id", "component_version_info")
        .style("position", "absolute")
        .style("width", "300px")
        .style("padding", "10px")
        .style("font", "12px sans-serif")
        .style("background", "lightgrey")
        .style("border-width", "1px")
        .style("border-style", "solid")
        .style("border-color", "grey")
        .style("border-radius", "8px")
        .style("pointer-events", "none")
        .style("opacity", 0);

    var tooltip = d3.select("body")
        .append("div")
        .style("position", "absolute")
        .style("width", "300px")
        .style("padding", "10px")
        .style("font", "12px sans-serif")
        .style("background", "lightgrey")
        .style("border-width", "1px")
        .style("border-style", "solid")
        .style("border-color", "grey")
        .style("border-radius", "8px")
        .style("pointer-events", "none")
        .style("opacity", 0);

    // 0 - ok
    // 1 - warn
    // 2 - crit
    function setHealth(element, state) {
        var element = d3Elem(element);
        // console.log('setHealth', element, state);

        var style = OK_STYLE;
        if(state == 1 || state == 'warn') {
            style = WARN_STYLE
        } else if(state == 2 || state == 'crit') {
            style = CRIT_STYLE
        }

        Object.keys(style).forEach(function(key) {
            var value = style[key];
            element.style(key, value)
        });
    }

    function getComponentIdByRect(rect) {
        var keys = Object.keys(componentRects);
        for (var i = 0; i < keys.length; i++) {
            if (rect.node() == componentRects[keys[i]].node()) {
                return keys[i]
            }
        }
        return undefined
    }

    function getComponentRect(component) {
        return componentRects[component]
    }

    function updateComponentState(component, state) {
        // console.log('updating component state', component, state);
        var rect = getComponentRect(component);
        if (rect == undefined) {
            // console.log('component rect not found');
            return
        }
        var hasWarns = state.jugglerWarns.length > 0;
        var hasCrits = state.jugglerCrits.length > 0;

        // console.log('hasWarns: ', hasWarns, 'hasCrits', hasCrits);

        if (hasCrits) {
            setHealth(rect, 'crit')
        } else if (hasWarns) {
            setHealth(rect, 'warn')
        } else {
            setHealth(rect, 'ok')
        }
    }

    function updateServiceMap() {
        // console.log('updating service map');
        Object.keys(clusterState).forEach(function(key) {
            updateComponentState(key, clusterState[key]);
        });
    }

    function updateClusterState() {
        findComponentRects();
        findComponentConnections();

        $.get('/z/monops-monitor.json', function (data) {
            // console.log('received new cluster state', data);
            var newClusterState = {};

            data.clusterState.applicationStates.forEach(function(appState) {
                newClusterState[appState.app.name] = appState
            });

            clusterState = newClusterState;
            updateServiceMap()
        })
    }

    function renderComponentDcInfo(qloudState) {
        var colorByDc = {
            "sas": "rgb(0, 255, 0)",
            "myt": "Thistle",
            "iva": "rgb(0, 255, 255)",
            "man": "rgb(255, 255, 0)",
            "vla": "Aquamarine",
        }

        html = "<table width='100%' height='100%' style='border-radius: 8px'><tr>"

        var cntByDc = qloudState.instancesByDc
        var widthPerDc = 100 / Object.keys(cntByDc).length

        for (var dc_i in Object.keys(colorByDc)) {
            var dc = Object.keys(colorByDc)[dc_i]
            if (dc in cntByDc) {
                html += "<td width='" + widthPerDc + "%' style='background-color:" + colorByDc[dc] + "'><center>"
                html += dc + "<br/>" + cntByDc[dc]
                html += "</center></td>"
            }
        }

        html += "</tr></table>"

        return html
    }

    function renderComponentVersionInfo(qloudState) {
            html = "<b>" + moment(qloudState.lastCreationDate).format("YYYY-MM-DD hh:mm") + "</b>"
                        + '<br/>'
                        + qloudState.lastComment
                        + '<br/><br/>'

            for (var component in Object.keys(qloudState.componentVersions)) {
                component = Object.keys(qloudState.componentVersions)[component]
                html += "<b>" + component + "</b></br>" + qloudState.componentVersions[component] + "<br/><br/>"
            }

            return html
    }

    function findComponentRects() {
        // console.log('start finding rects');
        // console.log('component rects: ', JSON.stringify(componentRects));
        d3.selectAll('text').each(function(d, i) {
            var el = d3.select(this);

            var x = getX(el);
            var y = getY(el);

            var componentName = this.textContent.toLowerCase();

            componentRects[componentName] = d3.select(findNearestRect(x, y, 'rect').node());
            var rect = componentRects[componentName]

            componentConnections[componentName] = {};

            var rectX = getX(rect)
            var rectWidth = getWidth(rect)

            var rectY = getY(rect)
            var rectHeight = getHeight(rect)

            // console.log('component rects: ', JSON.stringify(componentRects));

            function openComponenTab() {
                window.open("/z/monops-monitor/app?app=" + componentName)
            }

            var textDiv = d3.select(this.parentNode).select('div');

            el.on("click", openComponenTab);
            el.style('cursor', 'pointer');

            textDiv.on('click', openComponenTab);
            textDiv.style('cursor', 'pointer');

            textDiv
                .on("mouseover", function(d) {
                    rect = document.getElementById("map").getBoundingClientRect();

                    if(componentName in clusterState) {
                        var state = clusterState[componentName];

                        tooltip.transition()
                            .duration(200)
                            .style("opacity", .9);

                        var html = "";

                        state.jugglerCrits.concat(state.jugglerWarns).forEach(function(event) {
                            var style = event.status == "crit" ? "label-danger" : "label-warning";
                            html += "<span class=\"label " + style + "\">" + event.service + "</span><br/>"
                                + event.description.substring(0, 100)
                                + (event.description.length > 100 ? "..." : "")
                                + "<br/><br/>"
                        });

                        if (html == "") {
                            html = "<span style='color:green;'>OK</span>"
                        }

                        tooltip.html(html)
                            .style("left", (rect.x + rectX + rectWidth) + "px")
                            .style("top", (rect.y + rectY + rectHeight / 2) + "px");

                        if (state.qloudState) {
                            component_dc_info.transition()
                                        .duration(200)
                                        .style("opacity", .9);
                            component_version_info.transition()
                                        .duration(200)
                                        .style("opacity", 0.9);

                            component_dc_info.html(renderComponentDcInfo(state.qloudState))
                                        .style("width", rectWidth + "px")
                                        .style("height", "40px")
                                        .style("left", (rect.x +     rectX) + "px")
                                        .style("top", (rect.y + rectY - 42) + "px");

                            component_version_info.html(renderComponentVersionInfo(state.qloudState))
                                        .style("left", (rect.x + rectX - 300) + "px")
                                        .style("top", (rect.y + rectY + rectHeight / 2) + "px");
                        }
                    }
                })
                .on("mouseout", function(d) {
                    tooltip.transition()
                        .duration(500)
                        .style("opacity", 0);

                    component_dc_info.transition()
                        .duration(500)
                        .style("opacity", 0);

                    component_version_info.transition()
                        .duration(500)
                        .style("opacity", 0);
                });
        })
    }

    function findComponentConnections() {
        d3.selectAll('path').each(function(d, i) {
            var path_x1 = this.getPointAtLength(0).x;
            var path_y1 = this.getPointAtLength(0).y;

            var path_x2 = this.getPointAtLength(this.getTotalLength()).x;
            var path_y2 = this.getPointAtLength(this.getTotalLength()).y;

            var startRect = findNearestRect(path_x1, path_y1, 'rect');
            var endRect = findNearestRect(path_x2, path_y2, 'rect');

            var startId = getComponentIdByRect(startRect);
            var endId = getComponentIdByRect(endRect);

            if (startId != undefined && endId != undefined) {
                componentConnections[startId][endId] = d3.select(this);
                componentConnections[endId][startId] = d3.select(this);
            }
        })
    }

    // findComponentRects();
    // findComponentConnections();

    console.log('component rects: ', componentRects);
    console.log('component connections: ', componentConnections);

    updateClusterState();
//    var updateClusterStateIntervalId = setInterval(updateClusterState, 2500);
});

$(function () {
    $('#startTimeInput').datetimepicker({
        format:"YYYY-MM-DDTHH:mm:ss",
        minDate: moment(),
        sideBySide: true
    });
});
