package ru.yandex.chemodan.app.monops.admin;

import java.util.List;

import lombok.Data;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.monops.awacs.AwacsBalancerManager;
import ru.yandex.chemodan.app.monops.cluster.ApplicationInfo;
import ru.yandex.chemodan.app.monops.cluster.ApplicationState;
import ru.yandex.chemodan.app.monops.cluster.ClusterInfo;
import ru.yandex.chemodan.app.monops.cluster.ClusterState;
import ru.yandex.chemodan.app.monops.cluster.DashboardLink;
import ru.yandex.chemodan.app.monops.cluster.DelayedCloseDCZkRegistry;
import ru.yandex.chemodan.app.monops.cluster.InfraEventsZkRegistry;
import ru.yandex.chemodan.app.monops.cluster.JugglerSelector;
import ru.yandex.chemodan.app.monops.cluster.ManagedAppsZkRegistry;
import ru.yandex.chemodan.app.monops.cluster.MonopsQloudBinding;
import ru.yandex.chemodan.app.monops.cluster.UrlProvider;
import ru.yandex.chemodan.app.monops.cluster.UsefulLink;
import ru.yandex.chemodan.app.monops.worker.ClusterController;
import ru.yandex.chemodan.app.monops.worker.ClusterInfoRegistry;
import ru.yandex.chemodan.dc.closing.ClosedDataCenterInfo;
import ru.yandex.chemodan.dc.closing.ClosedDataCenterZkRegistry;
import ru.yandex.chemodan.util.auth.YateamAuthUtils;
import ru.yandex.chemodan.util.encrypt.HmacDigestUtil;
import ru.yandex.chemodan.util.exception.NotFoundException;
import ru.yandex.chemodan.util.web.OkPojo;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.commune.admin.z.ZRedirectException;
import ru.yandex.commune.alive2.AliveAppInfo;
import ru.yandex.commune.alive2.AliveAppsHolder;
import ru.yandex.inside.infra.InfraApiClient;
import ru.yandex.inside.infra.InfraEvent;
import ru.yandex.inside.infra.InfraEventSeverity;
import ru.yandex.inside.infra.InfraEventType;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.XmlRootElement;
import ru.yandex.misc.codec.Hex;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author tolmalev
 */
@ActionContainer
public class MonopsMainAdminPage {
    private static final Logger logger = LoggerFactory.getLogger(MonopsMainAdminPage.class);
    private static final ListF<String> ADMIN_LOGINS = Cf.list(
            "akinfold",
            "ignition",
            "ivanov-d-s",
            "kolyann",
            "vadzay",
            "bp2work"
    );

    private final ClusterController clusterController;
    private final ClusterInfoRegistry registry;
    private final ClosedDataCenterZkRegistry closedDataCenterZkRegistry;
    private final AliveAppsHolder aliveAppsHolder;
    private final ManagedAppsZkRegistry managedAppsZkRegistry;
    private final InfraEventsZkRegistry infraEventsZkRegistry;
    private final InfraApiClient infraApiClient;
    private final InfraEvent closeDcEventTemplate;
    private final InfraEvent closeDcDbsEventTemplate;
    private final DelayedCloseDCZkRegistry delayedCloseDCZkRegistry;
    private final AwacsBalancerManager awacsBalancerManager;

    public MonopsMainAdminPage(ClusterController clusterController, ClusterInfoRegistry registry,
                               ClosedDataCenterZkRegistry closedDataCenterZkRegistry, AliveAppsHolder aliveAppsHolder,
                               ManagedAppsZkRegistry managedAppsZkRegistry, InfraEventsZkRegistry infraEventsZkRegistry,
                               InfraApiClient infraApiClient, Integer infraServiceId,
                               Integer infraEnvironmentId,
                               DelayedCloseDCZkRegistry delayedCloseDCZkRegistry,
                               AwacsBalancerManager awacsBalancerManager)
    {
        this.clusterController = clusterController;
        this.registry = registry;
        this.closedDataCenterZkRegistry = closedDataCenterZkRegistry;
        this.aliveAppsHolder = aliveAppsHolder;
        this.managedAppsZkRegistry = managedAppsZkRegistry;
        this.infraEventsZkRegistry = infraEventsZkRegistry;
        this.infraApiClient = infraApiClient;
        this.delayedCloseDCZkRegistry = delayedCloseDCZkRegistry;
        this.awacsBalancerManager = awacsBalancerManager;
        this.closeDcEventTemplate = new InfraEvent(
                "Закрытие ДЦ от нагрузки",
                "ДЦ закрыт от внешнего и балансерного трафика",
                infraServiceId,
                infraEnvironmentId,
                InfraEventType.MAINTENANCE,
                InfraEventSeverity.MAJOR,
                true
        );
        this.closeDcDbsEventTemplate = new InfraEvent(this.closeDcEventTemplate);
        this.closeDcDbsEventTemplate.title = "Переключение мастеров баз из ДЦ";
        this.closeDcDbsEventTemplate.description = "Массовое переключение мастеров, возможны ошибки операций записи";
    }

    @ZAction(defaultAction = true, file = "monitor.xsl")
    @Path("/monops-monitor")
    public MonitorData monitor(@SpecialParam HttpServletRequestX reqX) {
        ListF<String> appNames = Cf.list("dataapi")
                .plus(
                        aliveAppsHolder.aliveApps()
                                .groupBy(AliveAppInfo::getAppName)
                                .mapValues(List::size)
                                .entries()
                                .sortedBy2Desc()
                                .get1()
                )
                .plus(
                        managedAppsZkRegistry.getAll()
                                .map(ManagedAppsZkRegistry.AppInfo::getId)
                                .sorted()
                );
        return new MonitorData(
                genToken(reqX),
                Cf.list("sas", "iva", "myt", "vla", "man"),
                Cf.toList(closedDataCenterZkRegistry.getAll()),
                Cf.toList(delayedCloseDCZkRegistry.getAll()),
                Cf.toList(infraEventsZkRegistry.getAll()),
                clusterController.getClusterInfo(),
                clusterController.getClusterState(),
                appNames
        );
    }

    @Path("/monops-monitor/open_force_closed_app")
    public OkPojo openForceClosedApp(
            @SpecialParam HttpServletRequestX reqX,
            @RequestParam("dcName") String dcName,
            @RequestParam("app") String app)
    {
        checkRights(reqX);
        closedDataCenterZkRegistry.openForceClosedApp(dcName, app);
        awacsBalancerManager.openAwacsBalancers(app, dcName);
        throw new ZRedirectException("/z/monops-monitor");
    }

    @Path("/monops-monitor/open_app_in_closed_dc")
    public OkPojo openAppInClosedDC(
            @SpecialParam HttpServletRequestX reqX,
            @RequestParam("dcName") String dcName,
            @RequestParam("app") String app)
    {
        checkRights(reqX);
        closedDataCenterZkRegistry.addExcludedApp(dcName, app);
        awacsBalancerManager.openAwacsBalancers(app, dcName);
        throw new ZRedirectException("/z/monops-monitor");
    }

    @Path("/monops-monitor/remove_app_exclude")
    public OkPojo removeAppExclude(
            @SpecialParam HttpServletRequestX reqX,
            @RequestParam("dcName") String dcName,
            @RequestParam("app") String app)
    {
        checkRights(reqX);
        awacsBalancerManager.closeAwacsBalancers(app, dcName);
        closedDataCenterZkRegistry.removeExcludedApp(dcName, app);
        throw new ZRedirectException("/z/monops-monitor");
    }

    @Path("/monops-monitor/force_close_app")
    public OkPojo forceCloseApp(
            @SpecialParam HttpServletRequestX reqX,
            @RequestParam("dcName") String dcName,
            @RequestParam("app") String app)
    {
        checkRights(reqX);
        awacsBalancerManager.closeAwacsBalancers(app, dcName);
        closedDataCenterZkRegistry.closeOneApp(dcName, app);
        throw new ZRedirectException("/z/monops-monitor");
    }

    @Path("/monops-monitor/open_dc")
    public OkPojo openDc(
            @SpecialParam HttpServletRequestX reqX,
            @RequestParam("dcName") String dcName)
    {
        checkRights(reqX);
        closedDataCenterZkRegistry.openDc(dcName);
        awacsBalancerManager.openAllBalancers(dcName);
        resolveInfraEvent(dcName, Option.of(Duration.standardMinutes(5)));
        throw new ZRedirectException("/z/monops-monitor");
    }

    @Path("/monops-monitor/open_dbs_dc")
    public OkPojo openDbsDc(
            @SpecialParam HttpServletRequestX reqX,
            @RequestParam("dcName") String dcName)
    {
        checkRights(reqX);
        closedDataCenterZkRegistry.openDbsDc(dcName);
        resolveInfraEvent(dcName, Option.empty());
        throw new ZRedirectException("/z/monops-monitor");
    }

    private void resolveInfraEvent(String dcName, Option<Duration> durationO) {
        infraEventsZkRegistry.getO(dcName).map(e -> {
            infraEventsZkRegistry.remove(dcName);
            return infraApiClient.resolveEvent(e.getId(), durationO);
        });
    }

    private void removeInfraEvent(String dcName) {
        infraEventsZkRegistry.getO(dcName).forEach(e -> {
            infraEventsZkRegistry.remove(dcName);
            infraApiClient.removeEvent(e.getId());
        });
    }

    private void createInfraEvent(InfraEvent eventTemplate, String dcName, String login, Instant startTime, Duration duration) {
        InfraEvent closeDcEvent = new InfraEvent(eventTemplate);
        closeDcEvent.startTime = startTime;
        closeDcEvent.finishTime = startTime.plus(duration);
        closeDcEvent.withDc(dcName, true);
        closeDcEvent.description += "\nинициатор: " + login + " через monops";
        InfraEvent event = infraApiClient.createEvent(closeDcEvent);
        infraEventsZkRegistry.put(new InfraEventsZkRegistry.DcEvent(event.id, dcName));
    }

    private void createInfraEvent(InfraEvent eventTemplate, String dcName, String login, Duration duration) {
        createInfraEvent(eventTemplate, dcName, login, Instant.now(), duration);
    }

    @Path(value = "/monops-monitor/close_dc", methods = HttpMethod.POST)
    public OkPojo closeDc(
            @SpecialParam HttpServletRequestX reqX,
            @RequestParam("dcName") String dcName,
            @RequestParam("isDelayed") Option<Boolean> isDelayed,
            @RequestParam("isDb") Boolean isDb,
            @RequestParam("startTime") String startTimeS)
    {
        checkRights(reqX);
        InfraEvent template;
        Duration duration;
        if (isDb) {
            template = closeDcDbsEventTemplate;
            duration = Duration.standardMinutes(30);
        } else {
            template = closeDcEventTemplate;
            duration = Duration.standardHours(2);
        }
        Validate.isEmpty(closedDataCenterZkRegistry.getAll().filter(ClosedDataCenterInfo::isClosed),
                "Нельзя закрывать несколько ДЦ одновременно!");

        if (startTimeS.indexOf("+") < 1) {
            startTimeS = startTimeS + "+0300";
        }
        Instant startTime = isDelayed.getOrElse(false) ? Instant.parse(startTimeS) : Instant.now();
        delayedCloseDCZkRegistry.scheduleCloseDc(dcName, startTime, isDb);
        createInfraEvent(template, dcName, getLogin(reqX), startTime, duration);
        throw new ZRedirectException("/z/monops-monitor");
    }

    @Path("/monops-monitor/cancel_delayed_close_dc")
    public OkPojo cancelDelayedCloseDc(
            @SpecialParam HttpServletRequestX reqX,
            @RequestParam("dcName") String dcName,
            @RequestParam("isDb") Boolean isDb)
    {
        checkRights(reqX);
        delayedCloseDCZkRegistry.cancelCloseDc(dcName, isDb);
        removeInfraEvent(dcName);
        throw new ZRedirectException("/z/monops-monitor");
    }

    private void checkRights(HttpServletRequestX reqX) {
        String token = reqX.getParameterO("token").getOrThrow("No CSRF token in request");
        String login = getLogin(reqX);
        checkToken(login, token);
        if (EnvironmentType.getActive() == EnvironmentType.PRODUCTION) {
            Validate.isTrue(ADMIN_LOGINS.containsTs(login), "Недостаточно прав");
        }
    }


    private String genToken(@SpecialParam HttpServletRequestX reqX) {
        return genToken(getLogin(reqX));
    }

    private String getLogin(HttpServletRequestX reqX) {
        return YateamAuthUtils.getLoginFromAttributeO(reqX)
                .orElse(Option.when(EnvironmentType.getActive() != EnvironmentType.PRODUCTION, "empty"))
                .getOrThrow("No login");
    }

    @ZAction(file = "app_info.xsl")
    @Path("/monops-monitor/app")
    public OneApplicationData appInfo(
            @SpecialParam HttpServletRequestX requestX,
            @RequestParam("app") String appName)
    {
        return new OneApplicationData(
                genToken(requestX),
                clusterController.getApplicationState(appName).getOrThrow(() -> new NotFoundException(appName)));
    }

    @Path(methods = HttpMethod.GET, value = "/monops-monitor/app/remove_qloud_binding")
    public void removeQloudBinding(
            @SpecialParam HttpServletRequestX reqX,
            @RequestParam("app") String appName)
    {
        ApplicationInfo applicationInfo = registry.get(appName);
        ApplicationInfo.ApplicationInfoBuilder builder = applicationInfo.toBuilder();

        builder.qloudBinding(Option.empty());

        registry.put(builder.build());

        throw new ZRedirectException("/monops-monitor/app?app=" + appName);
    }


    @Path(methods = HttpMethod.POST, value = "/monops-monitor/app/set_qloud_binding")
    public void addQloudBinding(
            @SpecialParam HttpServletRequestX reqX,
            @RequestParam("app") String appName,
            @RequestParam("application") String application,
            @RequestParam("environment") String environment,
            @RequestParam("component") String component)
    {
        ApplicationInfo applicationInfo = registry.get(appName);
        ApplicationInfo.ApplicationInfoBuilder builder = applicationInfo.toBuilder();

        builder.qloudBinding(Option.of(new MonopsQloudBinding(
                "disk",
                application,
                environment,
                Option.ofNullable(StringUtils.trimToNull(component))
        )));

        registry.put(builder.build());

        throw new ZRedirectException("/monops-monitor/app?app=" + appName);
    }

    @Path(methods = HttpMethod.POST, value = "/monops-monitor/app/add_juggler_selector")
    public void addJugglerSelector(
            @SpecialParam HttpServletRequestX reqX,
            @RequestParam("app") String appName,
            @RequestParam("host") String host,
            @RequestParam("service") String service)
    {
        JugglerSelector selector = new JugglerSelector(StringUtils.trim(host), StringUtils.trim(service));

        ApplicationInfo applicationInfo = registry.get(appName);
        ApplicationInfo.ApplicationInfoBuilder builder = applicationInfo.toBuilder();

        builder.jugglerSelectors(applicationInfo.getJugglerSelectors()
                .plus1(selector)
                .unique()
                .toList()
        );

        registry.put(builder.build());

        throw new ZRedirectException("/monops-monitor/app?app=" + appName);
    }

    @Path(methods = HttpMethod.GET, value = "/monops-monitor/app/remove_juggler_selector")
    public void removeJugglerSelector(
            @SpecialParam HttpServletRequestX reqX,
            @RequestParam("app") String appName,
            @RequestParam("host") String host,
            @RequestParam("service") String service)
    {
        JugglerSelector selector = new JugglerSelector(StringUtils.trim(host), StringUtils.trim(service));

        ApplicationInfo applicationInfo = registry.get(appName);
        ApplicationInfo.ApplicationInfoBuilder builder = applicationInfo.toBuilder();

        builder.jugglerSelectors(applicationInfo.getJugglerSelectors()
                .filter(s -> !s.equals(selector))
        );

        registry.put(builder.build());

        throw new ZRedirectException("/monops-monitor/app?app=" + appName);
    }

    @Path(methods = HttpMethod.POST, value = "/monops-monitor/app/add_alias")
    public void addAlias(
            @RequestParam("app") String appName,
            @RequestParam("url") String alias)
    {
        alias = StringUtils.trimToNull(alias);

        Validate.notNull(alias, "Empty alias");

        ApplicationInfo applicationInfo = registry.get(appName);

        ApplicationInfo.ApplicationInfoBuilder builder = applicationInfo.toBuilder();
        builder.aliases(applicationInfo.getAliases().plus1(alias).unique().toList());
        registry.put(builder.build());

        throw new ZRedirectException("/monops-monitor/app?app=" + appName);
    }

    @Path(methods = {HttpMethod.POST, HttpMethod.GET}, value = "/monops-monitor/app/remove_alias")
    public void removeAlias(
            @RequestParam("app") String appName,
            @RequestParam("url") String alias)
    {
        Validate.notNull(alias, "Empty alias");

        ApplicationInfo applicationInfo = registry.get(appName);

        ApplicationInfo.ApplicationInfoBuilder builder = applicationInfo.toBuilder();
        builder.aliases(applicationInfo.getAliases().filterNot(a -> a.equals(alias)));
        registry.put(builder.build());

        throw new ZRedirectException("/monops-monitor/app?app=" + appName);
    }

    @Path(methods = HttpMethod.POST, value = "/monops-monitor/app/add_link")
    public void addLink(
            @RequestParam("app") String appName,
            @RequestParam("url") String url,
            @RequestParam("title") String title,
            @RequestParam("linkType") String linkType)
    {
        url = StringUtils.trimToNull(url);
        title = StringUtils.trimToNull(title);

        Validate.notNull(url, "Empty url");
        Validate.notNull(title, "Empty title");

        ApplicationInfo applicationInfo = registry.get(appName);

        ApplicationInfo.ApplicationInfoBuilder builder = applicationInfo.toBuilder();

        switch (linkType) {
            case "useful":
                builder.usefulLinks(applicationInfo.getUsefulLinks().plus(new UsefulLink(url, Option.of(title))));
                break;
            case "dashboard":
                builder.dashboardLinks(
                        applicationInfo.getDashboardLinks().plus(new DashboardLink(url, Option.of(title))));
                break;
            default:
                throw new IllegalArgumentException("Unknown link type " + linkType);
        }

        registry.put(builder.build());

        throw new ZRedirectException("/monops-monitor/app?app=" + appName);
    }

    @Path(methods = HttpMethod.GET, value = "/monops-monitor/app/delete_link")
    public void deleteLink(
            @RequestParam("app") String appName,
            @RequestParam("url") String url,
            @RequestParam("linkType") String linkType)
    {
        ApplicationInfo applicationInfo = registry.get(appName);

        ApplicationInfo.ApplicationInfoBuilder builder = applicationInfo.toBuilder();

        switch (linkType) {
            case "useful":
                builder.usefulLinks(applicationInfo.getUsefulLinks().filter(l -> !url.equals(l.getUrl())));
                break;
            case "dashboard":
                builder.dashboardLinks(applicationInfo.getDashboardLinks().filter(l -> !url.equals(l.getUrl())));
                break;
            default:
                throw new IllegalArgumentException("Unknown link type " + linkType);
        }

        registry.put(builder.build());

        throw new ZRedirectException("/monops-monitor/app?app=" + appName);
    }

    @Path(methods = HttpMethod.POST, value = "/monops-monitor/app/add_awacs_balancer")
    public void addAwacsBalancer(
            @RequestParam("app") String appName,
            @RequestParam("balancer") String balancer)
    {
        balancer = StringUtils.trimToNull(balancer);
        Validate.notNull(balancer, "Empty balancer");

        ApplicationInfo applicationInfo = registry.get(appName);

        ApplicationInfo.ApplicationInfoBuilder builder = applicationInfo.toBuilder();
        builder.awacsBalancers(applicationInfo.getAwacsBalancers().plus1(balancer));
        registry.put(builder.build());

        throw new ZRedirectException("/monops-monitor/app?app=" + appName);
    }

    @Path(methods = HttpMethod.GET, value = "/monops-monitor/app/delete_awacs_balancer")
    public void deleteAwacsBalancer(
            @RequestParam("app") String appName,
            @RequestParam("balancer") String balancer)
    {
        ApplicationInfo applicationInfo = registry.get(appName);

        ApplicationInfo.ApplicationInfoBuilder builder = applicationInfo.toBuilder();
        builder.awacsBalancers(applicationInfo.getAwacsBalancers().filter(b -> !balancer.equals(b)));

        registry.put(builder.build());

        throw new ZRedirectException("/monops-monitor/app?app=" + appName);
    }

    @Path(methods = HttpMethod.GET, value = "/monops-monitor/app/move_link")
    public void moveLink(
            @RequestParam("app") String appName,
            @RequestParam("url") String url,
            @RequestParam("dir") int dir,
            @RequestParam("linkType") String linkType)
    {
        ApplicationInfo applicationInfo = registry.get(appName);

        ListF<UrlProvider> links;
        switch (linkType) {
            case "useful":
                links = Cf.toArrayList(applicationInfo.getUsefulLinks()).cast();
                break;
            case "dashboard":
                links = Cf.toArrayList(applicationInfo.getDashboardLinks()).cast();
                break;
            default:
                throw new IllegalArgumentException("Unknown link type " + linkType);
        }

        for (int i = 0; i < links.size(); i++) {
            if (url.equals(links.get(i).getUrl())) {
                int newIndex = i + dir;
                if (newIndex < 0 || newIndex >= links.size()) {
                    break;
                }

                UrlProvider link = links.remove(i);
                links.add(newIndex, link);

                ApplicationInfo.ApplicationInfoBuilder builder = applicationInfo.toBuilder();


                switch (linkType) {
                    case "useful":
                        builder.usefulLinks(links.cast());
                        break;
                    case "dashboard":
                        builder.dashboardLinks(links.cast());
                        break;
                    default:
                        throw new IllegalArgumentException("Unknown link type " + linkType);
                }

                registry.put(builder.build());

                break;
            }
        }

        throw new ZRedirectException("/monops-monitor/app?app=" + appName);
    }

    private String genToken(String login) {
        return genToken(login, Instant.now().getMillis());
    }

    private String genToken(String login, long timestamp) {
        return Hex.encode(HmacDigestUtil.sign(login + ":" + timestamp, "asdasf")) + ":" + timestamp;
    }

    public void checkToken(String login, String token) {
        String[] parts = token.split(":");
        Validate.equals(2, parts.length);

        Option<Long> timestampO = Cf.Long.parseSafe(parts[1]);
        Validate.some(timestampO);

        Instant now = Instant.now();
        Validate.ge(now.getMillis(), timestampO.get().longValue());

        Duration duration = new Duration(new Instant(timestampO.get()), now);
        Validate.le(duration, Duration.standardHours(24));

        if (!genToken(login, timestampO.get()).equals(token)) {
            throw new RuntimeException("Bad csrf token");
        }
    }

    @BenderBindAllFields
    @XmlRootElement(name = "content")
    @Data
    private static class MonitorData {
        private final String csrfToken;
        private final ListF<String> dcNames;
        private final ListF<ClosedDataCenterInfo> closedDataCenterInfo;
        private final ListF<DelayedCloseDCZkRegistry.DelayedCloseDC> delayedEvents;
        private final ListF<InfraEventsZkRegistry.DcEvent> dcInfraEvents;
        private final ClusterInfo clusterInfo;
        private final ClusterState clusterState;
        private final ListF<String> appNames;
    }

    @BenderBindAllFields
    @XmlRootElement(name = "content")
    @Data
    private static class OneApplicationData {
        private final String csrfToken;
        private final ApplicationState state;
    }
}
