package ru.yandex.chemodan.app.monops.awacs;

import java.util.Optional;
import java.util.function.Consumer;

import ru.yandex.chemodan.app.monops.awacs.entity.AwacsLocationWeights;
import ru.yandex.chemodan.app.monops.awacs.entity.AwacsSectionTypes;
import ru.yandex.chemodan.app.monops.awacs.entity.AwacsWeightValues;
import ru.yandex.chemodan.app.monops.awacs.entity.ItsVersion;
import ru.yandex.chemodan.app.monops.awacs.entity.ItsVersionUpdateRequest;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author friendlyevil
 */
public class AwacsBalancerController {
    private static final Logger log = LoggerFactory.getLogger(AwacsBalancerController.class);
    private final AwacsHttpClient awacsClient;

    public AwacsBalancerController(AwacsHttpClient awacsClient) {
        this.awacsClient = awacsClient;
    }

    public void closeBalancer(String balanceId, String dc) {
        updateBalancerWeights(balanceId, dc, location -> location.setWeight(0), false);
        log.info("Balancer {} was closed in {}", balanceId, dc);
    }

    public void openBalancer(String balanceId, String dc) {
        updateBalancerWeights(balanceId, dc, location -> location.setWeight(location.getDefaultWeight()), true);
        log.info("Balancer {} was opened in {}. Default weights was set", balanceId, dc);
    }

    private void updateBalancerWeights(String balanceId, String dc, Consumer<AwacsLocationWeights> updateWeightF,
                                       boolean updateAllLocations) {
        ValueWithEtag<AwacsWeightValues> weightValues = awacsClient.getWeightValues(balanceId);

        Optional<AwacsLocationWeights> awacsLocation = weightValues.getValue()
                .getSection(AwacsSectionTypes.BY_GEO)
                .flatMap(location -> location.getLocation(dc));
        if (!awacsLocation.isPresent()) {
            return;
        }

        if (!updateAllLocations) {
            updateWeightF.accept(awacsLocation.get());
        } else {
            weightValues.getValue()
                    .getSection(AwacsSectionTypes.BY_GEO).get()
                    .getLocations().values()
                    .forEach(updateWeightF);
        }

        ValueWithEtag<AwacsWeightValues> updatedWeights = awacsClient.updateWightValues(balanceId, weightValues);
        String newRevision = updatedWeights.getETag();

        ItsVersion currentItsValue = awacsClient.getCurrentItsValue(balanceId);

        ItsVersion itsVersion = awacsClient.updateItsValue(balanceId,
                new ItsVersionUpdateRequest(currentItsValue.getCurrentVersion(), newRevision));

        if (!newRevision.equals(itsVersion.getCurrentVersion())) {
            throw new ItsRevisionNotMatchException(itsVersion.getCurrentVersion(), newRevision);
        }
    }
}
