package ru.yandex.chemodan.app.monops.awacs;

import java.io.IOException;
import java.net.URI;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.util.EntityUtils;

import ru.yandex.chemodan.app.monops.awacs.entity.AwacsWeightValues;
import ru.yandex.chemodan.app.monops.awacs.entity.ItsVersion;
import ru.yandex.chemodan.app.monops.awacs.entity.ItsVersionUpdateRequest;
import ru.yandex.commune.json.jackson.ObjectMapperX;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.http.HttpHeaderNames;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author friendlyevil
 */
public class AwacsHttpClient {
    private static final Logger log = LoggerFactory.getLogger(AwacsHttpClient.class);
    private static final ObjectMapperX objectMapper = new ObjectMapperX(new ObjectMapper());

    private final HttpClient httpClient;
    private final String baseUrl;
    private final String oauthToken;

    public AwacsHttpClient(HttpClient httpClient, String baseUrl, String oauthToken) {
        this.httpClient = httpClient;
        this.baseUrl = baseUrl;
        this.oauthToken = oauthToken;
    }

    public ValueWithEtag<AwacsWeightValues> getWeightValues(String balancerId) {
        URI uri = UriBuilder.cons(baseUrl).appendPath(balancerId).appendPath("/weights/values/").build();
        HttpGet request = new HttpGet(uri);
        return executeAndGetEtag(request, getResponseHandler(AwacsWeightValues.class));
    }

    public ValueWithEtag<AwacsWeightValues> updateWightValues(String balancerId,
                                                              ValueWithEtag<AwacsWeightValues> weightValues) {
        URI uri = UriBuilder.cons(baseUrl).appendPath(balancerId).appendPath("/weights/values/").build();
        HttpPost request = new HttpPost(uri);
        request.setEntity(new StringEntity(objectMapper.writeValueAsString(weightValues.getValue()),
                ContentType.APPLICATION_JSON));
        request.addHeader(HttpHeaderNames.IF_MATCH, weightValues.getETag());
        return executeAndGetEtag(request, getResponseHandler(AwacsWeightValues.class));
    }

    public ItsVersion getCurrentItsValue(String balancerId) {
        URI uri = UriBuilder.cons(baseUrl).appendPath(balancerId).appendPath("/weights/its_value/").build();
        return execute(new HttpGet(uri), getResponseHandler(ItsVersion.class));
    }

    public ItsVersion updateItsValue(String balancerId, ItsVersionUpdateRequest request) {
        URI uri = UriBuilder.cons(baseUrl).appendPath(balancerId).appendPath("/weights/its_value/").build();
        HttpPost post = new HttpPost(uri);
        post.setEntity(new StringEntity(objectMapper.writeValueAsString(request), ContentType.APPLICATION_JSON));
        return execute(post, getResponseHandler(ItsVersion.class));
    }

    private <T> ValueWithEtag<T> executeAndGetEtag(HttpUriRequest request,
                                                   ResponseHandler<? extends T> responseHandler) {
        return execute(request, r -> new ValueWithEtag<>(responseHandler.handleResponse(r), getEtagFromResponse(r)));
    }

    private <T> T execute(HttpUriRequest request, ResponseHandler<? extends T> responseHandler) {
        request.addHeader("Authorization", "OAuth " + oauthToken);

        HttpResponse response = null;
        try {
            response = httpClient.execute(request);
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode < 200 || statusCode >= 300) {
                throw new AwacsBadStatusCode(statusCode);
            }

            return responseHandler.handleResponse(response);
        } catch (IOException e) {
            log.error("Failed request to its", e);
            throw ExceptionUtils.translate(e);
        } finally {
            if (response != null) {
                EntityUtils.consumeQuietly(response.getEntity());
            }
        }
    }

    private String getEtagFromResponse(HttpResponse response) {
        Header eTag = response.getFirstHeader(HttpHeaderNames.ETAG);
        if (eTag == null || eTag.getValue() == null) {
            throw new EtagNotFoundException();
        }

        return StringUtils.strip(eTag.getValue(), "\"");
    }

    private static <T> ResponseHandler<? extends T> getResponseHandler(Class<T> clazz) {
        return r -> objectMapper.readValue(r.getEntity().getContent(), clazz);
    }
}
