package ru.yandex.chemodan.app.monops.cluster;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.inside.qloud.client.QloudApiClient;
import ru.yandex.inside.qloud.client.QloudComponentRuntime;
import ru.yandex.inside.qloud.client.QloudEnvironmentState;
import ru.yandex.inside.qloud.client.QloudProject;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author tolmalev
 */
public class QloudStateCacheManager extends DelayingWorkerServiceBeanSupport {

    private static final Logger logger = LoggerFactory.getLogger(QloudStateCacheManager.class);

    private final QloudApiClient qloudApiClient;

    private ClusterInfo clusterInfo = ClusterInfo.EMPTY;
    private QloudProject projectState = new QloudProject("disk", Cf.list(), Cf.map());

    private MapF<MonopsQloudBinding, Tuple2<QloudEnvironmentState, MapF<String, QloudComponentRuntime>>> cachedStates = Cf.concurrentHashMap();

    private final ExecutorService executorService = Executors.newFixedThreadPool(10);

    public QloudStateCacheManager(QloudApiClient qloudApiClient) {
        this.qloudApiClient = qloudApiClient;
    }

    @Override
    protected Duration defaultDelay() {
        return Duration.standardMinutes(1);
    }

    @Override
    protected void execute() {
        projectState = qloudApiClient.getProject("disk");

        ListF<MonopsQloudBinding> qloudBindings =
                clusterInfo.applications.flatMap(ApplicationInfo::getQloudBinding);

        ListF<? extends Future<?>> futures = qloudBindings.map(binding -> {
            return executorService.submit(() -> {
                QloudEnvironmentState envState =
                        executeWithLongRetries(() -> qloudApiClient.getEnvironmentState(binding.project, binding.application, binding.environment));

                long envVersion = envState.version;

                if (binding.component.isPresent()) {
                    envState = new QloudEnvironmentState(
                            envState.objectId,
                            envState.name,
                            envState.projectName,
                            envState.applicationName,
                            envState.version,
                            envState.status,
                            envState.author,
                            envState.comment,
                            envState.creationDate,
                            envState.statusMessage,
                            envState.domains,
                            envState.userVariables,
                            envState.components.filterKeys(key -> key.equals(binding.component.get())),
                            envState.versions);
                }

                if (!cachedStates.getO(binding).isPresent() || !cachedStates.get(binding)._1.equals(envState)) {
                    MapF<String, QloudComponentRuntime> componentRuntimes = envState.components.values()
                            .map(component -> executeWithLongRetries(() -> qloudApiClient.getComponentRuntime(component.objectId, envVersion)))
                            .toMapMappingToKey(c -> c.name);

                    cachedStates.put(binding, Tuple2.tuple(envState, componentRuntimes));
                }
            });
        });

        futures.forEach(future -> {
            try {
                future.get();
            } catch (Exception e) {
                logger.warn("Failed to update state from qloud: {}", e);
            }
        });
    }

    private <T> T executeWithLongRetries(Callable<T> callable) {
        int retries = 0;
        while (true) {
            try {
                return callable.call();
            } catch (Exception e) {
                if (retries > 60) {
                    throw ExceptionUtils.translate(e);
                }
                retries++;
                ThreadUtils.sleep(Duration.standardSeconds(1));
            }
        }
    }

    public Option<Tuple2<QloudEnvironmentState, MapF<String, QloudComponentRuntime>>> getState(MonopsQloudBinding binding) {
        return cachedStates.getO(binding);
    }

    public QloudProject getProjectState() {
        return projectState;
    }

    public void setClusterInfo(ClusterInfo clusterInfo) {
        this.clusterInfo = clusterInfo;
    }
}
