package ru.yandex.chemodan.app.monops.worker;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import com.fasterxml.jackson.databind.JsonNode;
import org.apache.http.HttpResponse;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.monops.cluster.ApplicationInfo;
import ru.yandex.chemodan.app.monops.cluster.ApplicationMainMetricsState;
import ru.yandex.chemodan.app.monops.cluster.ApplicationMetricInfo;
import ru.yandex.chemodan.app.monops.cluster.ApplicationState;
import ru.yandex.chemodan.app.monops.cluster.ClusterInfo;
import ru.yandex.chemodan.app.monops.cluster.ClusterState;
import ru.yandex.chemodan.app.monops.cluster.ExtendedConnectionsInfo;
import ru.yandex.chemodan.app.monops.cluster.ExtendedHttpComponentConnection;
import ru.yandex.chemodan.app.monops.cluster.ExtendedJdbcComponentConnection;
import ru.yandex.chemodan.app.monops.cluster.JugglerEvent;
import ru.yandex.chemodan.app.monops.cluster.JugglerSelector;
import ru.yandex.chemodan.app.monops.cluster.QloudCompactState;
import ru.yandex.chemodan.app.monops.cluster.QloudStateCacheManager;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.chemodan.util.servicemap.HttpComponentConnection;
import ru.yandex.chemodan.util.servicemap.JdbcComponentConnection;
import ru.yandex.chemodan.util.servicemap.ServiceMap;
import ru.yandex.inside.qloud.client.QloudComponentRuntime;
import ru.yandex.inside.qloud.client.QloudEnvironmentState;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.parse.BenderJsonParser;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.Abstract200ExtendedResponseHandler;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author tolmalev
 */
public class ClusterController extends DelayingWorkerServiceBeanSupport {
    private static final BenderJsonParser<GraphiteResponseItem> graphiteItemParser =
            Bender.jsonParser(GraphiteResponseItem.class);

    private ClusterInfo clusterInfo = ClusterInfo.EMPTY;
    private ClusterState clusterState = ClusterState.EMPTY;

    private final ExecutorService executorService = Executors.newFixedThreadPool(30);

    private final QloudStateCacheManager qloudStateCacheManager;
    private final ComponentConnectionsManager componentConnectionsManager;

    public ClusterController(QloudStateCacheManager qloudStateCacheManager, ComponentConnectionsManager componentConnectionsManager) {
        this.qloudStateCacheManager = qloudStateCacheManager;
        this.componentConnectionsManager = componentConnectionsManager;

        this.setSleepBeforeFirstRun(false);
        this.setDelay(Duration.standardMinutes(1));
    }

    public ClusterState getClusterState() {
        return clusterState;
    }

    public void setClusterInfo(ClusterInfo clusterInfo) {
        this.clusterInfo = clusterInfo;
    }

    @Override
    protected void execute() throws Exception {
        ListF<ApplicationState> applicationStates = clusterInfo.applications
                .map(app -> executorService.submit(() -> getAppState(app)))
                .map(applicationStateFuture -> {
                    try {
                        return applicationStateFuture.get();
                    } catch (Exception e) {
                        throw ExceptionUtils.translate(e);
                    }
                });

        clusterState = new ClusterState(applicationStates);
    }

    private ApplicationState getAppState(ApplicationInfo app) {
        ListF<JugglerEvent> jugglerEvents;
        try {
            jugglerEvents = getJugglerEvents(app.getJugglerSelectors());
        } catch (RuntimeException e) {
            jugglerEvents = Cf.list(new JugglerEvent(
                    "Failed to get juggler state: " + ExceptionUtils.getAllMessages(e),
                    "monops_juggler",
                    "get_events_" + app.getName(),
                    JugglerEvent.Status.CRIT
            ));
        }

        ApplicationMainMetricsState mainMetrics = new ApplicationMainMetricsState(Cf.list());
        try {
            mainMetrics = getMainMetrics(app);
        } catch (RuntimeException e) {
            jugglerEvents = jugglerEvents.plus1(new JugglerEvent(
                    "Failed to get metrics: " + ExceptionUtils.getAllMessages(e),
                    "monops_metrics",
                    "get_metrics_" + app.getName(),
                    JugglerEvent.Status.CRIT
            ));
        }

        Option<Tuple2<QloudEnvironmentState, MapF<String, QloudComponentRuntime>>> qloudState =
                app.getQloudBinding().flatMapO(qloudStateCacheManager::getState);

        Option<ServiceMap> serviceMap = componentConnectionsManager.getO(app.name);

        return new ApplicationState(app,
                jugglerEvents,
                mainMetrics,
                qloudState.map(t -> new QloudCompactState(t._1, t._2)),
                serviceMap.map(this::resolveConnections)
        );
    }

    public ExtendedConnectionsInfo resolveConnections(ServiceMap serviceMap) {
        return new ExtendedConnectionsInfo(
                serviceMap.httpConnections.map(this::resolveHttpConnection),
                serviceMap.jdbcConnections.map(this::resolveJdbcConnection)
        );
    }

    public ExtendedHttpComponentConnection resolveHttpConnection(HttpComponentConnection connection) {
        Option<String> resolved = connection.hostname.flatMapO(hostName -> clusterInfo.getApplications().find(app ->
                app.getJugglerSelectors().exists(js -> js.getHost().isSome(hostName))
                || app.getAliases().containsTs(hostName)
        )).map(ApplicationInfo::getName);

        return new ExtendedHttpComponentConnection(resolved, connection);
    }

    public ExtendedJdbcComponentConnection resolveJdbcConnection(JdbcComponentConnection component) {
        return new ExtendedJdbcComponentConnection(Option.empty(), component);
    }

    private ApplicationMainMetricsState getMainMetrics(ApplicationInfo app) {
        ListF<ApplicationMetricInfo> graphiteMetrics = app.getMainMetrics()
                .getMetrics()
                .filter(m -> m.type == ApplicationMetricInfo.Type.GR);

        ListF<String> graphiteTargets = graphiteMetrics
                .map(metric -> {
                    return "alias(transformNull(movingAverage(" + metric.expression + ", '1min'), 0), '" + metric.title + "')";
                });

        if (graphiteTargets.size() == 0) {
            return new ApplicationMainMetricsState(Cf.list());
        }

        String url = UrlUtils.addParameters("https://gr-mg.yandex-team.ru/render/",
                graphiteTargets
                        .zipWith(i -> "target")
                        .invert()
                        .plus1(new Tuple2("from", "-5minutes"))
                        .plus1(new Tuple2("format", "json"))
        );

        HttpClient httpClient = makeHttpClient();

        ListF<GraphiteResponseItem> items = ApacheHttpClientUtils
                .execute(new HttpGet(url), httpClient, new Abstract200ExtendedResponseHandler<ListF<GraphiteResponseItem>>() {
                    @Override
                    protected ListF<GraphiteResponseItem> handle200Response(HttpResponse response)
                            throws ClientProtocolException, IOException
                    {
                        InputStreamSource iss = InputStreamSourceUtils.wrap(response.getEntity().getContent());
                        return graphiteItemParser.parseListJson(iss);
                    }
                });

        MapF<String, Double> nameToValue = items.toMap(i -> {
            return Tuple2.tuple(i.target, i.datapoints.last().first());
        });

        return new ApplicationMainMetricsState(graphiteMetrics.map(m -> new ApplicationMainMetricsState.MetricState(
                m,
                m.getTitle(),
                nameToValue.getOrElse(m.getTitle(), 0.0)
        )));
    }

    private ListF<JugglerEvent> getJugglerEvents(ListF<JugglerSelector> jugglerSelectors) {
        HttpClient httpClient = makeHttpClient();

        return jugglerSelectors.flatMap(s -> {
            return getJugglerEvents(httpClient, s, "CRIT").plus(getJugglerEvents(httpClient, s, "WARN"));
        });
    }

    private HttpClient makeHttpClient() {
        return ApacheHttpClientUtils.Builder
                .create()
                .singleThreaded()
                .withHttpsSupport(ApacheHttpClientUtils.HttpsSupport.TRUST_ALL)
                .withTimeout(new Timeout(30000, 1000))
                .build();
    }

    private ListF<JugglerEvent> getJugglerEvents(HttpClient httpClient, JugglerSelector s, String status) {
        UriBuilder uri = UriBuilder.cons("https://juggler-api.search.yandex.net/api/events/complete_events");

        uri.addParamO("host_name", s.host);
        uri.addParamO("service_name", s.service);

        uri.addParam("status", status);
        uri.addParam("do", "1");

        return ApacheHttpClientUtils
                .execute(new HttpGet(uri.build()), httpClient, new Abstract200ExtendedResponseHandler<ListF<JugglerEvent>>() {
            @Override
            protected ListF<JugglerEvent> handle200Response(HttpResponse response)
                    throws ClientProtocolException, IOException
            {
                ListF<JugglerEvent> result = Cf.arrayList();

                InputStreamSource iss = InputStreamSourceUtils.wrap(response.getEntity().getContent());
                JsonNode node = JsonNodeUtils.getNode(iss.readText());

                for (Map.Entry<String, JsonNode> t2 : Cf.x(node.fields()).toList()) {
                    String host = t2.getKey();
                    for (Map.Entry<String, JsonNode> entry : Cf.x(t2.getValue().fields()).toList()) {
                        String service = entry.getKey();
                        JsonNode eventNode = entry.getValue();

                        result.add(new JugglerEvent(
                                eventNode.get("description").get(0).get("description").asText(),
                                host,
                                service,
                                JugglerEvent.Status.valueOf(eventNode.get("status").get(0).asText())
                        ));
                    }
                }
                return result;
            }
        });
    }

    public ClusterInfo getClusterInfo() {
        return clusterInfo;
    }

    public Option<ApplicationInfo> getApplicationInfo(String appName) {
        return clusterInfo.applications.find(app -> app.getName().equals(appName));
    }

    public Option<ApplicationState> getApplicationState(String appName) {
        return getClusterState().applicationStates
                .find(app -> app.app.getName().equals(appName))
                .map(app -> app.toBuilder()
                        .app(clusterInfo.applications.find(a -> a.getName().equals(appName)).get())
                        .build()
                );
    }

    @BenderBindAllFields
    static class GraphiteResponseItem {
        final String target;
        final ListF<ListF<Double>> datapoints;

        GraphiteResponseItem(String target, ListF<ListF<Double>> datapoints) {
            this.target = target;
            this.datapoints = datapoints;
        }
    }
}
