package ru.yandex.chemodan.app.monops.worker;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.monops.awacs.AwacsBalancerManager;
import ru.yandex.chemodan.app.monops.awacs.AwacsControllerContextConfiguration;
import ru.yandex.chemodan.app.monops.cluster.ApplicationInfo;
import ru.yandex.chemodan.app.monops.cluster.ApplicationMainMetrics;
import ru.yandex.chemodan.app.monops.cluster.ApplicationMetricInfo;
import ru.yandex.chemodan.app.monops.cluster.ClusterInfo;
import ru.yandex.chemodan.app.monops.cluster.DashboardLink;
import ru.yandex.chemodan.app.monops.cluster.DelayedCloseDCZkRegistry;
import ru.yandex.chemodan.app.monops.cluster.InfraEventsZkRegistry;
import ru.yandex.chemodan.app.monops.cluster.JugglerSelector;
import ru.yandex.chemodan.app.monops.cluster.ManagedAppsZkRegistry;
import ru.yandex.chemodan.app.monops.cluster.MonopsQloudBinding;
import ru.yandex.chemodan.app.monops.cluster.QloudStateCacheManager;
import ru.yandex.chemodan.app.monops.cluster.UsefulLink;
import ru.yandex.chemodan.boot.value.OverridableValuePrefix;
import ru.yandex.chemodan.dc.closing.ClosedDataCenterZkRegistry;
import ru.yandex.chemodan.util.http.HttpClientConfigurator;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.client.ZkManager;
import ru.yandex.inside.qloud.client.QloudApiClient;

/**
 * @author tolmalev
 */
@Configuration
@Import({AwacsControllerContextConfiguration.class})
public class ClusterControllerContextConfiguration {
    @Autowired
    private ZkManager zkManager;

    @Qualifier("zkAllRoot")
    @Autowired
    private ZkPath zkRoot;

    @Bean
    public ClusterController clusterController(QloudStateCacheManager qloudStateCacheManager, ComponentConnectionsManager componentConnectionsManager) {
        return new ClusterController(qloudStateCacheManager, componentConnectionsManager);
    }

    @Bean
    @OverridableValuePrefix("qloud_api")
    public HttpClientConfigurator qloudApiHttpClientConfigurator() {
        return new HttpClientConfigurator();
    }

    @Bean
    public QloudApiClient qloudApiClient(@Value("${qloud-api.token:-fake_token}") String token) {
        return new QloudApiClient("https://qloud-ext.yandex-team.ru/api/v1/", token, qloudApiHttpClientConfigurator().configure());
    }

    @Bean
    public ManagedAppsZkRegistry managedAppsZkRegistry() {
        ManagedAppsZkRegistry registry = new ManagedAppsZkRegistry(zkRoot.child("monops").child("managed-apps"));
        zkManager.addClient(registry);
        return registry;
    }

    @Bean
    public InfraEventsZkRegistry infraEventsZkRegistry() {
        InfraEventsZkRegistry registry = new InfraEventsZkRegistry(zkRoot.child("monops").child("infra-events"));
        zkManager.addClient(registry);
        return registry;
    }

    @Bean
    public DelayedCloseController delayedCloseController(ClosedDataCenterZkRegistry closedDataCenterZkRegistry,
                                                         DelayedCloseDCZkRegistry delayedCloseDCZkRegistry,
                                                         AwacsBalancerManager awacsBalancerManager) {
        return new DelayedCloseController(closedDataCenterZkRegistry, delayedCloseDCZkRegistry, awacsBalancerManager);
    }

    @Bean
    public DelayedCloseDCZkRegistry delayedCloseDCZkRegistry() {
        DelayedCloseDCZkRegistry registry = new DelayedCloseDCZkRegistry(zkRoot.child("monops").child("delayed-close-dc"));
        zkManager.addClient(registry);
        return registry;
    }

    @Bean
    public ComponentConnectionsManager componentConnectionsManager(QloudStateCacheManager qloudStateCacheManager, ManagedAppsZkRegistry managedAppsZkRegistry) {
        return new ComponentConnectionsManager(qloudStateCacheManager, managedAppsZkRegistry);
    }

    @Bean
    public QloudStateCacheManager qloudStateCacheManager(QloudApiClient qloudApiClient) {
        return new QloudStateCacheManager(qloudApiClient);
    }

    @Bean
    public ClusterInfoRegistry clusterInfoRegistry(ClusterController clusterController, QloudStateCacheManager qloudStateCacheManager,
                                                   ComponentConnectionsManager componentConnectionsManager)
    {
        ClusterInfoRegistry registry = new ClusterInfoRegistry(zkRoot.child("monops").child("cluster-info"));

        registry.addListener(state -> {
            ClusterInfo clusterInfo = new ClusterInfo(state.toList());

            clusterController.setClusterInfo(clusterInfo);
            qloudStateCacheManager.setClusterInfo(clusterInfo);
            componentConnectionsManager.setClusterInfo(clusterInfo);
        });

        zkManager.addClient(registry);
        return registry;
    }

//    @PostConstruct
    public void initServiceMap(ClusterInfoRegistry clusterInfoRegistry) {
        Cf.list(
                ApplicationInfo
                        .builder()
                        .name("dataapi")
                        .dashboardLinks(Cf.list(
                                new DashboardLink(
                                        "https://grafana.yandex-team.ru/dashboard/db/disk_dataapi",
                                        Option.of("Основной дашборд в графане")
                                ),
                                new DashboardLink(
                                        "https://yasm.yandex-team.ru/template/panel/disk_dataapi/",
                                        Option.of("Дашборд в головане")
                                )
                        ))
                        .usefulLinks(Cf.list(
                                new UsefulLink("https://wiki.yandex-team.ru/disk/personality/", "Generic api"),
                                new UsefulLink("https://wiki.yandex-team.ru/disk/java/watch/datasync/", "Инструкция для дежурного")
                        ))
                        .mainMetrics(ApplicationMainMetrics.builder().metrics(Cf.list(
                                new ApplicationMetricInfo(
                                        ApplicationMetricInfo.Type.GR,
                                        "2xx",
                                        "media.disk.combaine.dataapi_disk_yandex_net.dataapi_nginx.2xx"
                                ),
                                new ApplicationMetricInfo(
                                        ApplicationMetricInfo.Type.GR,
                                        "5xx",
                                        "media.disk.combaine.dataapi_disk_yandex_net.dataapi_nginx.5xx"
                                )
                        )).build())
                        .build(),
                ApplicationInfo
                        .builder()
                        .name("uploader")
                        .dashboardLinks(Cf.list(
                                new DashboardLink(
                                        "https://grafana.yandex-team.ru/dashboard/db/disk_uploader",
                                        Option.of("Основной дашборд в графане")
                                ),
                                new DashboardLink(
                                        "https://yasm.yandex-team.ru/template/panel/disk_uploader/",
                                        Option.of("Дашборд в головане")
                                )
                        ))
                        .usefulLinks(Cf.list(
                                new UsefulLink("https://wiki.yandex-team.ru/disk/java/watch/uploader/", "Инструкция для дежурного")
                        ))
                        .mainMetrics(ApplicationMainMetrics.builder().metrics(Cf.list(
                                new ApplicationMetricInfo(
                                        ApplicationMetricInfo.Type.GR,
                                        "not uploaded to mulca",
                                        "sumSeries(media.disk.disk_uploader.*.uploader-sensors.uploader_sensors_queue_incomplete_and_uploaded_locally_and_not_uploaded_to_mulca)"
                                ),
                                new ApplicationMetricInfo(
                                        ApplicationMetricInfo.Type.GR,
                                        "waiting for user",
                                        "sumSeries(media.disk.disk_uploader.*.uploader-sensors.uploader_sensors_queue_incomplete_requests_waiting_for_user_count)"
                                )
                        )).build())
                        .build(),
                ApplicationInfo
                        .builder()
                        .name("webdav")
                        .dashboardLinks(Cf.list(
                                new DashboardLink(
                                        "https://yasm.yandex-team.ru/template/panel/disk_webdav-java/",
                                        Option.of("Дашборд в головане")
                                )
                        ))
                        .usefulLinks(Cf.list(
                                new UsefulLink("https://wiki.yandex-team.ru/disk/java/watch/webdav/", "Инструкция для дежурного")
                        ))
                        .mainMetrics(ApplicationMainMetrics.builder().metrics(Cf.list(
                                new ApplicationMetricInfo(
                                        ApplicationMetricInfo.Type.YASM,
                                        "2xx",
                                        "signals=unistat-webdav_access_2xx_ammm;hosts=ASEARCH;itype=deploy;stage=disk-webdav-stable"
                                ),
                                new ApplicationMetricInfo(
                                        ApplicationMetricInfo.Type.YASM,
                                        "5xx - 507",
                                        "signals=diff(unistat-webdav_access_5xx_ammm,unistat-webdav_access_507_ammm);hosts=ASEARCH;itype=deploy;stage=disk-webdav-stable"
                                )
                        )).build())
                        .build(),
                ApplicationInfo
                        .builder()
                        .name("front")
                        .jugglerSelectors(Cf.list(
                                new JugglerSelector("disk.yandex.ru"),
                                new JugglerSelector("disk.yandex.ru-devops")
                        ))
                        .build(),
                new ApplicationInfo("apidb"),
                new ApplicationInfo("smcdb"),
                new ApplicationInfo("dataapi_worker"),
                new ApplicationInfo("smartcache-client"),
                new ApplicationInfo("smartcache-worker"),
                new ApplicationInfo("lenta-loader"),
                new ApplicationInfo("lenta-worker"),
                new ApplicationInfo("notifier"),
                ApplicationInfo.builder()
                        .name("streaming")
                        .qloudBinding(Option.of(new MonopsQloudBinding(
                                "disk",
                                "disk-videostreaming",
                                "production",
                                Option.empty()
                        )))
                        .build(),
                new ApplicationInfo("downloader"),

                new ApplicationInfo("intapi"),
                ApplicationInfo
                        .builder()
                        .name("cloud-api")
                        .jugglerSelectors(Cf.list(
                                new JugglerSelector("cloud-api.yandex.net"),
                                new JugglerSelector("cloud-api.yandex-team.ru"),
                                new JugglerSelector("disk_api")
                        ))
                        .build(),

                ApplicationInfo
                        .builder()
                        .name("mpfs")
                        .dashboardLinks(Cf.list(
                                new DashboardLink(
                                        "https://grafana.yandex-team.ru/dashboard/db/disk_mpfs",
                                        Option.of("Основной дашборд в графане")
                                ),
                                new DashboardLink(
                                        "https://yasm.yandex-team.ru/template/panel/disk_mpfs/",
                                        Option.of("Дашборд в головане")
                                )
                        ))
                        .usefulLinks(Cf.list(
                                new UsefulLink("https://wiki.yandex-team.ru/disk/mpfs/", "Главная вики страница"),
                                new UsefulLink("https://wiki.yandex-team.ru/disk/mpfs/api/json/", "Описание API"),
                                new UsefulLink("https://wiki.yandex-team.ru/disk/mpfs/meps/", "MEPs")
                        ))
                        .mainMetrics(ApplicationMainMetrics.builder().metrics(Cf.list(
                                new ApplicationMetricInfo(
                                        ApplicationMetricInfo.Type.GR,
                                        "2xx",
                                        "media.disk.combaine.mpfs_disk_yandex_net.mpfs_nginx.2xx"
                                ),
                                new ApplicationMetricInfo(
                                        ApplicationMetricInfo.Type.GR,
                                        "5xx",
                                        "media.disk.combaine.mpfs_disk_yandex_net.mpfs_nginx.5xx"
                                )
                        )).build())
                        .build(),

                new ApplicationInfo("mworker"),
                new ApplicationInfo("queller"),
                new ApplicationInfo("mqueue"),

                new ApplicationInfo("dv_backend")
        ).forEach(clusterInfoRegistry::put);
    }
}
