package ru.yandex.chemodan.app.monops.worker;

import java.io.IOException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.monops.cluster.ApplicationInfo;
import ru.yandex.chemodan.app.monops.cluster.ClusterInfo;
import ru.yandex.chemodan.app.monops.cluster.ManagedAppsZkRegistry;
import ru.yandex.chemodan.app.monops.cluster.QloudStateCacheManager;
import ru.yandex.chemodan.util.servicemap.ServiceMap;
import ru.yandex.inside.qloud.client.QloudComponentRuntime;
import ru.yandex.inside.qloud.client.QloudEnvironmentState;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.apache.v4.Abstract200ExtendedResponseHandler;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author tolmalev
 */
public class ComponentConnectionsManager extends DelayingWorkerServiceBeanSupport {
    private static final Logger logger = LoggerFactory.getLogger(ComponentConnectionsManager.class);

    private final QloudStateCacheManager qloudStateCacheManager;
    private final ManagedAppsZkRegistry managedAppsZkRegistry;
    private ClusterInfo clusterInfo;

    private final ExecutorService executorService = Executors.newFixedThreadPool(10);
    private final MapF<String, ServiceMap> serviceMapByComponents = Cf.concurrentHashMap();

    public ComponentConnectionsManager(QloudStateCacheManager qloudStateCacheManager, ManagedAppsZkRegistry managedAppsZkRegistry) {
        this.qloudStateCacheManager = qloudStateCacheManager;
        this.managedAppsZkRegistry = managedAppsZkRegistry;
    }

    @Override
    protected Duration defaultDelay() {
        return Duration.standardMinutes(1);
    }

    @Override
    protected void execute() {
        clusterInfo.getApplications()
            .filter(app -> app.getQloudBinding().isPresent())
            .forEach(app -> {
                String applicationName = app.getQloudBinding().get().getApplication();
                boolean isManagedAppsComponent = managedAppsZkRegistry.getO(applicationName).isPresent();
                if (!isManagedAppsComponent) {
                    executorService.submit(() -> updateComponentConnections(app));
                }
            });
    }

    public void updateComponentConnections(ApplicationInfo app) {
        Option<Tuple2<QloudEnvironmentState, MapF<String, QloudComponentRuntime>>> stateO =
                qloudStateCacheManager.getState(app.getQloudBinding().get());

        stateO.forEach(t2 -> {
            ListF<String> allHosts = t2._1.components.values()
                    .flatMap(component -> component.runningInstances)
                    .flatMap(i -> i.host);

            ListF<String> instancesPart = Random2.R.shuffle(allHosts).take(5);

            for (String instance : instancesPart) {
                boolean allIs404 = true;

                for (int port : Cf.list(80, 81)) {
                    try {
                        logger.debug("Try to get component connections for app={} from {}", app.name, instance);
                        HttpGet get = new HttpGet("http://" + instance + ":" + port + "/m/service_map");
                        ServiceMap serviceMap = ApacheHttpClientUtils.execute(get, new Abstract200ExtendedResponseHandler<ServiceMap>() {
                            @Override
                            protected ServiceMap handle200Response(HttpResponse response) throws IOException {
                                return Bender.jsonParser(ServiceMap.class).parseJson(InputStreamSourceUtils.wrap(response.getEntity().getContent()));
                            }
                        }, new Timeout(5000, 5000));

                        serviceMapByComponents.put(app.getName(), serviceMap);

                        return;
                    } catch (HttpException e) {
                        if (!e.getStatusCode().isSome(404)) {
                            allIs404 = false;
                            logger.error("Failed to get service map from {} for {}: {}", instance, app.name, e);
                        }
                    } catch (Exception e) {
                        allIs404 = false;
                        logger.error("Failed to get service map from {} for {}: {}", instance, app.name, e);
                    }
                }
                if (allIs404) {
                    logger.debug("Service map action isn't supported by {}", app.name);
                    return;
                }
            }
        });
    }

    public void setClusterInfo(ClusterInfo clusterInfo) {
        this.clusterInfo = clusterInfo;
    }

    public Option<ServiceMap> getO(String appName) {
        return serviceMapByComponents.getO(appName);
    }
}
