# -*- coding: utf-8 -*-
"""
Добавление в очередь модерации публичных ссылок на файлы
https://st.yandex-team.ru/CHEMODAN-73521
Как добавлять объекты в очередь модерации:
1. На вход в параметр csv-filepath передаем csv файл с полями resource_id и child_porn с разделителем ";"
2. В параметре env указываем тип среды из списка ['test', 'prod']
3. После запуска скрипта проверить по ссылке https://disk-support.yandex-team.ru/moderation_queue, что объекты добавлены
"""
from argparse import ArgumentParser
import csv
from datetime import datetime
import urllib
import mpfs.engine.process
mpfs.engine.process.setup_admin_script()
from mpfs.core.filesystem.dao.legacy import CollectionRoutedDatabase
from mpfs.core.factory import get_resource_by_file_id, get_resource_by_address
from mpfs.common.errors import ResourceNotFound

db = CollectionRoutedDatabase()


def get_short_url(uid, resource):

    if resource.is_fully_public():
        return resource.get_short_url()

    cur_address = resource.address
    names = [cur_address.name]
    adress_parents = cur_address.get_parents()
    for address in adress_parents[:-1]:
        parent_resource = get_resource_by_address(uid, address)
        if parent_resource.is_fully_public():
            path = urllib.quote('/'.join(reversed(names)).encode('utf-8'))
            url = '%s/%s' % (parent_resource.get_short_url(), path)
            return url
        names.append(address.name)


def insert_to_queue(source, url, link_type, description, hid):
    moderation_item = {
        "created": datetime.utcnow(),
        "source": source,
        "links": [{"type": link_type, "url": url}],
        "status": "not-moderated",
        "hid": hid,
        "description": description
    }
    db.support_moderation_queue.insert(moderation_item)


def main(csv_filepath, env):
    with open(csv_filepath) as f:
        reader = csv.DictReader(f, delimiter=';', quoting=csv.QUOTE_MINIMAL)
        for row in reader:
            resource_id = row['resource_id']
            cp_coef = row['child_porn']
            uid, file_id = resource_id.split(':')

            try:
                resource = get_resource_by_file_id(uid, file_id)
                hid = resource.hid
                short_url = get_short_url(uid, resource)
                if short_url:
                    if env == 'test':
                        short_url = short_url.replace('https://yadi.sk', 'https://public.dst.yadi.sk')
                    insert_to_queue(source="child_porn", url=short_url, link_type="file", description=cp_coef, hid=hid)
                else:
                    print('resource %s is not public' % resource_id)
            except ResourceNotFound:
                print('resource %s is not found' % resource_id)


if __name__ == '__main__':
    parser = ArgumentParser()
    parser.add_argument('--csv-filepath', dest='csv_filepath', type=str, required=True)
    parser.add_argument('--env', dest='env', choices=['test', 'prod'], required=True)
    args = parser.parse_args()

    main(args.csv_filepath, args.env)
