# -*- coding: utf-8 -*-
u"""
Выставления флага IS_PAID_WITHOUT_PS_BILLING юзерам по наличию платных услуг.

Для https://st.yandex-team.ru/CHEMODAN-76530

Как запускать:
# Выставить флаг всем юзерам у кого есть платные услуги.
sudo python set_is_paid_without_ps_billing.py
# Запуск без записи флага в базу, но с полным выводом в логи.
sudo python set_is_paid_without_ps_billing.py --dry-run
# Выставить флаг всем юзерам у кого есть платные услуги и у котрых uid оканчивается цифрой 0
sudo python set_is_paid_without_ps_billing.py -s 0
# Выставить флаг всем юзерам у кого есть платные услуги и у котрых uid оканчивается цифрой 1
sudo python set_is_paid_without_ps_billing.py -s 1
# Выставить флаг всем юзерам у кого есть платные услуги и у котрых uid оканчивается цифрами 01
sudo python set_is_paid_without_ps_billing.py -s 01
# Выставить флаг всем юзерам у кого есть платные услуги и у котрых uid оканчивается цифрами 10
sudo python set_is_paid_without_ps_billing.py -s 10
"""
import sys
from argparse import ArgumentParser

import mpfs.engine.process
mpfs.engine.process.setup_admin_script()

from mpfs.core.user.base import User
from mpfs.core.user.constants import IS_PAID_WITHOUT_PS_BILLING
from mpfs.common.static.tags.billing import PID, UID
from mpfs.core.billing.product.catalog import PS_BILLING_PRODUCT_IDS
from mpfs.core.billing.service import ServiceList
from mpfs.core.user.standart import PAID_PRODUCTS_IDS


DEFAULT_LOG_PREFIX = 'SYNC_IS_PAID_WITHOUT_PS_BILLING'


default_log = mpfs.engine.process.get_default_log()


def log(msg):
    print(msg)
    message = u'%s: %s' % (DEFAULT_LOG_PREFIX, msg)
    default_log.info(message)


def main(uid_suffix, dry_run):
    mpfs_billing_paid_uids = set()
    services = ServiceList(product=list(PAID_PRODUCTS_IDS))
    services_count = len(services)
    log('found %s services with product_ids in [%s]' % (services_count, ', '.join(PAID_PRODUCTS_IDS)))
    for service in services:
        product_id = service.get(PID)
        uid = service.get(UID)
        if uid_suffix and not uid.endswith(uid_suffix):
            continue
        if (product_id not in PS_BILLING_PRODUCT_IDS and uid not in mpfs_billing_paid_uids):
            mpfs_billing_paid_uids.add(uid)

    mpfs_billing_paid_uids_count = len(mpfs_billing_paid_uids)

    log('selected %s users to set IS_PAID_WITHOUT_PS_BILLING setting.' % mpfs_billing_paid_uids_count)

    for uid in mpfs_billing_paid_uids:
        if not dry_run:
            User(uid).set_setting(IS_PAID_WITHOUT_PS_BILLING, True)
        log('set %s for uid %s' % (IS_PAID_WITHOUT_PS_BILLING, uid))


if __name__ == '__main__':
    log(' '.join(sys.argv))
    parser = ArgumentParser(__doc__)
    parser.add_argument('-s', dest='uid_suffix', type=str, help=u'суффикс uid-ов к которым применить скрипт')
    parser.add_argument('--dry-run', dest='dry_run', action='store_true')
    args = parser.parse_args()

    uid_suffix = args.uid_suffix
    if uid_suffix and not uid_suffix.isdigit():
        raise ValueError('-s value must consists of digits')

    main(uid_suffix, args.dry_run)
