# -*- coding: utf-8 -*-
"""
Remove file id collisions
https://st.yandex-team.ru/DISKSUP-17130
"""
import mpfs.engine.process
mpfs.engine.process.setup_admin_script()


from argparse import ArgumentParser
import sys

from mpfs.dao.session import Session

from mpfs.metastorage.postgres.queries import SqlTemplatedQuery
from mpfs.core.filesystem.dao.file import FileDAO, FileDAOItem
from mpfs.core.filesystem.dao.folder import FolderDAO, FolderDAOItem
from mpfs.core.filesystem.resources.base import Resource


SQL_FOLDERS_BY_FILE_ID = SqlTemplatedQuery("""
SELECT
    %(SELECT_FOLDER_FIELDS_TEMPLATE)s,
    (SELECT path FROM code.fid_to_path(d.fid, d.uid)) as path
FROM
    disk.folders d
WHERE
    d.uid=:uid AND
    d.id=:file_id
""")

SQL_FILES_BY_FILE_ID = SqlTemplatedQuery("""
SELECT
    %(SELECT_FILE_FIELDS_TEMPLATE)s,
    (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
FROM
    disk.files f JOIN disk.storage_files s USING (storage_id)
WHERE
    f.uid=:uid AND
    f.id=:file_id
""")

LOG_TEMPLATE = u'REMOVE_FILE_ID_COLLISIONS: uid %s path %s changed from %s to %s'

default_log = mpfs.engine.process.get_default_log()


def uncollide(uid, file_ids):
    session = Session.create_from_uid(uid)

    for line in file_ids:
        file_id = line[:-1]
        if not file_id:
            continue

        params = {
            'uid': FolderDAOItem.convert_mongo_value_to_postgres_for_key('uid', uid)[1],
            'file_id': FolderDAOItem.convert_mongo_value_to_postgres_for_key('data.file_id', file_id)[1],
        }

        folders = map(lambda row: FolderDAOItem.create_from_pg_data(row),
                      session.execute(SQL_FOLDERS_BY_FILE_ID, params).fetchall())
        files = map(lambda row: FileDAOItem.create_from_pg_data(row),
                    session.execute(SQL_FILES_BY_FILE_ID, params).fetchall())

        items = folders + files

        for item in items[1:]:
            generated_file_id = Resource.generate_file_id(uid, item.path)

            default_log.info(LOG_TEMPLATE % (uid, item.path, item.file_id, generated_file_id))

            if isinstance(item, FolderDAOItem):
                session.execute('UPDATE disk.folders SET id=:id WHERE uid=:uid AND fid=:fid', {
                    'uid': FolderDAOItem.convert_mongo_value_to_postgres_for_key('uid', uid)[1],
                    'fid': FolderDAOItem.convert_mongo_value_to_postgres_for_key('_id', item.fid)[1],
                    'id': FolderDAOItem.convert_mongo_value_to_postgres_for_key('data.file_id', generated_file_id)[1]
                })
            else:
                session.execute('UPDATE disk.files SET id=:id WHERE uid=:uid AND fid=:fid', {
                    'uid': FileDAOItem.convert_mongo_value_to_postgres_for_key('uid', uid)[1],
                    'fid': FileDAOItem.convert_mongo_value_to_postgres_for_key('_id', item.fid)[1],
                    'id': FileDAOItem.convert_mongo_value_to_postgres_for_key('data.file_id', generated_file_id)[1]
                })


if __name__ == '__main__':
    parser = ArgumentParser()
    parser.add_argument('-u', dest='uid', type=str, required=True)
    args = parser.parse_args()

    stream = sys.stdin

    uncollide(args.uid, stream)
