#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""Админский скрипт, проставляющий etime фотосрезным файлам.

Проверяет наличие etime у фотосрезных файлов. В случае отсутствия проверяет наличие DateTimeOriginal в выводе
regenerate-exif. Если DateTimeOriginal есть, то проставляет его в качестве etime фотосрезному файлу.

    :Example:

    $ python mpfs-admin-ensure-etime.py --uid 4000756161
    $ python mpfs-admin-ensure-etime.py --uid 4000756161 --dry-run

    ..warning:: Скрипт требует админских привелегий.

"""

import logging

from argparse import ArgumentParser
from datetime import datetime

from mpfs.engine.process import setup_admin_script
# Call before other imports
setup_admin_script()

from mpfs.common.util import say
from mpfs.core.address import Address
from mpfs.core.bus import Bus
from mpfs.core.filesystem.resources.base import PHOTO_MIME_TYPES
from mpfs.core.services.kladun_service import Kladun
from mpfs.core.services.smartcache_service import SmartcacheService
from mpfs.metastorage.mongo.collections.all_user_data import AllUserDataCollection


VERBOSE = True
DRY_RUN = False
TOTAL_FOUND = 0


def update_photoslice(uid):
    """Обновить снепшот фотосреза для `uid`

    :type uid: str
    :rtype: None
    """
    if DRY_RUN:
        return

    smartcache_client = SmartcacheService()
    try:
        smartcache_client.notify_worker(uid)
        say('Photoslice has been successfully updated', logging.INFO, VERBOSE)
    except Exception:
        say('Unable to update photoslice:', logging.ERROR, VERBOSE, exc_info=True)


def _seconds_from_datetimeoriginal(date):
    """Сконвертировать строку DateTimeOriginal в миллисекунды от начала эпохи

    .. warning:: exif информация содержит наивное время в московской временной зоне.
        Приходится явно его конвертировать при расчете количества секунд.

    .. warning:: В документации сказано, что формат даты и времени фиксированный и может
        содержать временную зону. На деле встретились два формата.
        Временную зону не поддерживаем.

    .. seealso::
        http://www.sno.phy.queensu.ca/~phil/exiftool/faq.html  п.5

    :type date: str
    :rtype: int
    """

    for fmt in ('%Y/%m/%d %H:%M:%S', '%Y:%m:%d %H:%M:%S'):
        try:
            d = datetime.strptime(date, fmt)
            break
        except ValueError:
            pass
    else:
        # ни один формат не подошел или неправильная дата
        # например 0000:00:00 00:00:00 если даты в файле нет
        # @metal: "Кладун, в таком случае, игнорирует etime в мете при покладке"
        raise

    epoch = datetime.utcfromtimestamp(0)
    # питон думает, что офсет равен 3
    localized_epoch = epoch.replace(hour=4)

    return int((d - localized_epoch).total_seconds())


def main(uid):
    """Найти фотосрезные файлы пользователя, у которых
    отсутсвует `etime` и проставить его.

    :type uid: str
    :rtype: None
    """

    global TOTAL_FOUND

    kladun_client = Kladun()
    fs = Bus()
    for db_result in AllUserDataCollection().find_on_uid_shard(uid, {'uid': uid, 'type': 'file'}, None, False, timeout=False):
        doc = db_result.record
        path = doc['key']
        file_mid = doc['data']['meta']['file_mid']
        mimetype = doc['data'].get('mimetype', '').encode('utf-8')

        if mimetype in PHOTO_MIME_TYPES and 'etime' not in doc['data']['meta']:

            _, exif = kladun_client.extract_exif(file_mid)
            if u'DateTimeOriginal' in exif:
                address = Address.Make(uid, path)

                try:
                    etime = _seconds_from_datetimeoriginal(exif[u'DateTimeOriginal'])
                    say('Found missing `etime` in : %s' % address.id, verbose=VERBOSE)
                except ValueError as e:
                    say('Unable to read `etime` for: %s, reason: %s' % (address.id, e), logging.ERROR, verbose=VERBOSE)
                    continue

                say('Fixing `etime` for: %s' % address.id, verbose=VERBOSE)
                if not DRY_RUN:
                    fs.patch_file(uid, address.id, changes={'etime': etime})
                    say('Fixed `etime` for: %s' % address.id, verbose=VERBOSE)

                TOTAL_FOUND += 1


if __name__ == '__main__':
    parser = ArgumentParser(description='Set missing `etime` for photoslice files')

    parser.add_argument('-d', '--dry-run', action='store_true', help='do not make any changes')
    parser.add_argument('-u', '--uid', required=True, help='`uid` to search for photoslice files for')

    args = parser.parse_args()

    if args.dry_run:
        DRY_RUN = True

    try:
        main(args.uid)
        if TOTAL_FOUND:
            update_photoslice(args.uid)
    except Exception as e:
        if TOTAL_FOUND:
            update_photoslice(args.uid)
        say('', level=logging.ERROR, verbose=VERBOSE, exc_info=True)
        raise
