#!/usr/bin/python
# -*- coding: utf-8 -*-
"""

Скрипт чистки юзера в монгосе/на шардах

"""
import sys
import traceback
import time
import os
import signal

from multiprocessing import Process, Queue, Array
from optparse import OptionParser, Option

import mpfs.engine.process
mpfs.engine.process.setup_admin_script()

from mpfs.common.errors import StorageInitUser
from mpfs.core.user.base import User
from mpfs.metastorage.mongo.sharded.migrator import MongoMigrator
from mpfs.metastorage.mongo.sharded.migrator import MigrationIdlenessTimeout, MigrationFatalError
from mpfs.metastorage.mongo.sharded.migrator import MigrationTemporaryError
from mpfs.metastorage.mongo.collections.meta import MPFSMongoCollectionsMeta

status_log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

usage = "usage: sudo -u nginx /usr/sbin/%prog -h"

option_list = (
    Option(
        '-u', '--uid',
        action='store',
        dest='uid',
        type='string',
        help='user uid'
    ),
    Option(
        '--src',
        action='store',
        dest='src',
        type='string',
        help='source: where to clean, mongos or shard id (mongos by default)',
        default='mongos'
    ),
    Option(
        '-c', '--collection',
        action='store',
        dest='collection',
        type='string',
        help='purge only one collection if specified',
        default=None,
    ),
    Option(
        '-d', '--dry',
        action='store_true',
        dest='dry',
        help='do not perform real purge, only show amount of data to be purged',
        default=False,
    ),
    Option(
        '-v', '--verbose',
        action='store_true',
        dest='verbose',
        help='verbose everything (False by default)',
        default=False,
    ),
    Option(
        '-p', '--processes',
        action='store',
        dest='processes',
        type='int',
        help='amount of processes (one process per collection)',
        default=None,
    ),
    Option(
        '--configdb',
        action='store',
        dest='configdb',
        type='string',
        help='config db (localhost:27017/config by default)',
        default='localhost:27017/config'
    ),
    Option(
        '-w', '--write',
        action='store',
        dest='w',
        type='int',
        help='write concern (default 2)',
        default=2,
    ),
)

parser = OptionParser(usage, option_list=option_list)
(options, args) = parser.parse_args()


def say(something):
    status_log.info(something)
    if options.verbose:
        print(something)


def _worker_method(**kwargs):
    _id = kwargs['_id']
    parent = kwargs['parent']
    queue = kwargs['queue']
    src = kwargs['src']
    uid = kwargs['uid']
    migrator = kwargs['migrator']

    # shared arrays
    processed = kwargs['processed']
    success = kwargs['success']
    failed = kwargs['failed']

    say("%s:%s started to purge..." % (os.getpid(), parent))

    while True:
        if os.getppid() != parent:
            say("%s:%s finished due to parent death" % (os.getpid(), parent))
            sys.exit()

        if queue.empty():
            return

        try:
            collection = queue.get(timeout=1)
        except Queue.Empty:
            return

        processed[_id] += 1

        try:
            migrator.purge(uid, src, check=True, collections=[collection])
        except Exception:
            failed[_id] += 1
            error_log.error("%s:%s failed on exception for UID:%s" % (os.getpid(), parent, uid))

        success[_id] += 1


def run_multiprocessing_purge(uid, src, migrator, collections):
    super_parent_pid = os.getppid()
    parent_pid = os.getpid()

    workers = []
    processes_amount = options.processes if options.processes else len(collections)
    failed_array = Array('i', range(processes_amount))
    processed_array = Array('i', range(processes_amount))
    success_array = Array('i', range(processes_amount))

    def killall():
        if os.getpid() == parent_pid:
            say('parent process %s was killed, killing workers' % parent_pid)
            for item in workers:
                worker_pid = item['worker'].pid
                worker_id = item['_id']
                os.kill(worker_pid, signal.SIGKILL)
                say('%s:%s terminated' % (worker_pid, parent_pid))
            sys.exit(1)
        else:
            process_pid = os.getpid()
            say('%s:%s was killed' % (process_pid, parent_pid))
            os.kill(process_pid, signal.SIGKILL)

    #  навешиваем обработчика SIGTERM
    def sigterm(signum, frame):
        killall()
    signal.signal(signal.SIGTERM, sigterm)

    try:
        queue = Queue()

        for collection in collections:
            queue.put(collection)

        for i in xrange(processes_amount):
            # обязательно инициируем начальные значения массивов синхронизации
            failed_array[i] = 0
            processed_array[i] = 0
            success_array[i] = 0

            kwargs = {
                '_id': i,
                'queue': queue,
                'parent': parent_pid,
                'src': src,
                'uid': uid,
                'migrator': migrator,
                'failed': failed_array,
                'processed': processed_array,
                'success': success_array
            }

            worker = Process(
                target=_worker_method,
                kwargs=kwargs
            )

            workers.append({'_id': i, 'worker': worker, 'pid': None, 'finished': False})

        # запускаем процессы
        for worker in workers:
            worker['worker'].start()
            worker['pid'] = worker['worker'].pid
    except Exception, e:
        error_log.error(traceback.format_exc())
        raise e

    # смотрим за процессами
    while True:
        all_workers_finished = True

        if os.getppid() != super_parent_pid:
            say("%s finished due to super-parent death" % os.getpid())
            killall()

        for worker in workers:
            worker['worker'].join(timeout=1)
            if not worker['worker'].is_alive():
                worker['finished'] = True
                if worker['worker'].exitcode != 0:
                    failed_array[worker['_id']] += 1

        for worker in workers:
            if not worker['finished']:
                all_workers_finished = False

        if all_workers_finished:
            break

    # финально проверяем флаги по процессам
    for i in xrange(processes_amount):
        pid = workers[i]['pid']
        if failed_array[i] > 0:
            error_log.error('%s:%s failed flag %s times' % (pid, parent_pid, failed_array[i]))
            raise MigrationFatalError('something failed in subprocesses, watch error-tskv.log')

        if processed_array[i] != success_array[i]:
            error_log.error('%s:%s took %s collections, processed %s' % (pid, parent_pid, processed_array[i], success_array[i]))
            raise MigrationFatalError('something failed in subprocesses, watch error-tskv.log')


def extra_checks(uid):
    """
    Extra checks before purge old user's data

    https://st.yandex-team.ru/CHEMODAN-26251
    """
    shard_name = mpfs.engine.process.usrctl().info(uid)['shard'] or 'mongos'

    if shard_name == 'mongos':
        mongos_db = mpfs.engine.process.dbctl().database()
        user_data_coll = mongos_db['user_data']
        user_index_coll = mongos_db['user_index']
    else:
        mapper = mpfs.engine.process.dbctl().mapper
        shard_connection = mapper.rspool.get_connection_for_rs_name(shard_name)
        user_data_coll = shard_connection.user_data['user_data']
        user_index_coll = shard_connection.user_index['user_index']

    if user_data_coll.find({'uid': uid}).count() <= 0:
        say('No docs in "user_data" at "%s" for uid "%s"' % (shard_name, uid))
        sys.exit(1)

    if user_index_coll.find({'_id': uid}).count() <= 0:
        say('No docs in "user_index" at "%s" for uid "%s"' % (shard_name, uid))
        sys.exit(1)


def main(uid, src, collections):
    mapper = mpfs.engine.process.dbctl().mapper
    all_shards = mapper.rspool.get_all_shards_names()
    all_shards.append('mongos')
    current_shard = mapper.get_rsname_for_uid(uid) or 'mongos'

    try:
        User(uid)
    except StorageInitUser:
        say('user %s does not exist' % uid)
        sys.exit(1)

    if src not in all_shards:
        say('bad purge shard "%s"' % src)
        sys.exit(1)

    if src == current_shard:
        say('bad source shard "%s", user %s lives right there' % (src, uid))
        sys.exit(1)

    extra_checks(uid)

    try:
        mpfs.engine.process.reset_cached()
        say('STARTING PURGE %s on %s' % (uid, src))
        start_time = time.time()

        migrator = MongoMigrator(
            dry=options.dry,
            silent=not options.verbose,
            w=options.w,
            status_log=status_log,
            error_log=error_log
        )

        # удаляем данные
        if not options.dry:
            run_multiprocessing_purge(uid, src, migrator, collections)
    except MigrationTemporaryError, e:
        say("EXCEPTION: %s" % e)
        error_log.error(traceback.format_exc())
        sys.exit(50)
    except MigrationFatalError, e:
        say("EXCEPTION: %s" % e)
        error_log.error(traceback.format_exc())
        sys.exit(1)
    except MigrationIdlenessTimeout, e:
        say("EXCEPTION: %s" % e)
        error_log.error(traceback.format_exc())
        sys.exit(50)
    except Exception, e:
        say("EXCEPTION: %s" % e)
        error_log.error(traceback.format_exc())
        sys.exit(1)

    end_time = time.time()
    say('FINISHED PURGING %s on %s' % (uid, src))
    say('TOTAL ELAPSED: %.3f sec' % (end_time - start_time))

if __name__ == "__main__":
    if not (options.uid and options.src):
        parser.print_help()
        sys.exit(0)

    if not options.collection:
        coll_meta = MPFSMongoCollectionsMeta()
        coll_meta.setup()
        collections = coll_meta.get_sharded_collections()
    else:
        collections = [options.collection]

    main(
        options.uid,
        options.src,
        collections,
    )
