#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
В STDOUT msg=<имя конфликта>\tuid=<uid>[\tsid=<sid>][\tpid=<pid>][\toid=<order_number>][\tstate=<состояние сервиса>]

Всем измененным или созданным сущностям в базе добавляем поле 'description': <имя скрипта> <время запуска>
"""
import calendar
import datetime
import multiprocessing
import optparse
import re
from dateutil.relativedelta import relativedelta

from mpfs.engine.process import setup_admin_script
setup_admin_script()

from mpfs.common.errors.billing import (
    BigBillingBadResult,
    BillingOrderNotFound,
)
from mpfs.common.static.tags.billing import (
    AMOUNT,
    ATTRIBUTES,
    BONUS,
    MONTH,
    PERIOD,
)
from mpfs.common.util import (
    iterdbuids,
    ctimestamp,
)
from mpfs.core.billing import api as billing_api
from mpfs.core.billing import (
    Product,
    Service,
)
from mpfs.core.billing.client import Client
from mpfs.core.billing.order import (
    ArchiveOrder,
    Order,
)
from mpfs.core.billing.order.subscription import Subscription
from mpfs.core.billing.processing import billing
from mpfs.core.billing.processing.pushthelimits import (
    PRODUCT_APP_INSTALL_ID,
    PRODUCT_FILE_UPLOADED_ID,
    PRODUCT_PROMO_SHARED_ID,
    PRODUCT_INITIAL_10GB_ID,
    PRODUCT_INITIAL_3GB_ID
)
from mpfs.core.billing.product.catalog import PRIMARY_PRODUCTS
from mpfs.core.filesystem.quota import Quota
from mpfs.core.metastorage.control import (
    billing_orders,
    billing_orders_history,
    billing_services,
    billing_subscriptions,
    user_index
)
from mpfs.core.billing.processing.repair import repair_services
from mpfs.core.services.billing_service import BB
import mpfs.engine.process

init_services_old = {PRODUCT_INITIAL_3GB_ID, PRODUCT_APP_INSTALL_ID, PRODUCT_PROMO_SHARED_ID, PRODUCT_FILE_UPLOADED_ID}
init_services_new = {PRODUCT_INITIAL_10GB_ID}

# Статусы подписок; 0 - идет триальный период, способ оплаты не указан; 1 - идет триальный период, способ оплаты указан;
# 3 - оплачена; https://wiki.yandex-team.ru/Balance/Simple/XMLRPC/#balancesimple.checkorder
ACTIVE_SUB_BB_STATES = {0, 1, 3}
HOST = '127.0.0.1'
# Считаем, что 10 минут разницы между заказами в биллингах - это много
TIME_SPAN = 600
PRODUCT_IDS = [x['id'] for x in PRIMARY_PRODUCTS]
PRODUCT_DATA = {x['id']: x for x in PRIMARY_PRODUCTS}

BB_TYPE_SUB = 'subs'
BB_TYPE_APP = 'app'

BB_BTIME = 'subs_until_ts_msec'
BB_LBTIME = 'payment_ts_msec'

COMMON_FUCKUP_PID = 'common_fuckup'

mpfs.engine.process.setup_logs(default_log_in='mpfs')
log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


def format_log(msg, **kwargs):
    kv = ['msg=' + msg] + ['%s=%s' % (x[0], x[1]) for x in kwargs.iteritems()]
    return ' '.join(kv)

class BBOrderInfo(object):

    def __init__(self, bb_record):
        parsed_values = self.parse_csv(bb_record)
        self.product_type = parsed_values['product_type']
        self.pid = parsed_values['pid']
        self.ctime = parsed_values['ctime']
        self.lbtime = parsed_values['lbtime']
        self.oid = parsed_values['oid']
        if parsed_values['subs_until_ts_msec']:
            self.btime = BBOrderInfo.msec_to_sec(parsed_values['subs_until_ts_msec'])
        else:
            # счиатем btime в unix time
            end_dt = (datetime.datetime.fromtimestamp(int(self.lbtime)) +
                      relativedelta(months=PRODUCT_DATA[self.pid][PERIOD][MONTH]))
            self.btime = calendar.timegm(end_dt.timetuple())
        if self.product_type == BB_TYPE_SUB:
            self.auto = 1
        else:
            self.auto = 0
        self.subs_state = None
        self.uid = parsed_values['uid']
        self.payment_method = None

    @staticmethod
    def parse_csv(csv):
        """
        Вид данных BB в csv
        ("ORDER_ID","PRODUCT_TYPE","SERVICE_PRODUCT","CREATOR_UID","ORDER_TS_MSEC","SUBS_UNTIL_TS_MSEC",
        "PAYMENT_TS_MSEC")
        """
        fields = csv.strip().split(',')
        order_id = fields[0]
        product_type = fields[1]
        service_product = fields[2]
        uid = fields[3]
        order_ts_msec = fields[4]
        subs_until_ts_msec = fields[5]
        payment_ts_msec = fields[6]
        bb_pid = service_product.strip('"')
        res = dict()
        res['product_type'] = product_type.strip('"')
        res['pid'] = re.sub('_' + res['product_type'], '', bb_pid)
        res['ctime'] = BBOrderInfo.msec_to_sec(order_ts_msec)
        res['lbtime'] = BBOrderInfo.msec_to_sec(payment_ts_msec)
        res['subs_until_ts_msec'] = subs_until_ts_msec
        res['oid'] = order_id
        res['uid'] = uid
        return res

    @staticmethod
    def msec_to_sec(bb_time):
        return int(bb_time) / 1000

    @classmethod
    def create_from_bb_order(cls, bb_order, oid, uid):
        bb_record = ','.join([oid, bb_order['product_type'], bb_order['service_product_id'],
                              uid, bb_order['order_ts_msec'], bb_order.get('subs_until_ts_msec', ''),
                              bb_order['payment_ts_msec']])
        result = cls(bb_record)
        result.payment_method = bb_order['payment_method_type']
        if result.product_type == BB_TYPE_SUB:
            result.subs_state = bb_order['subs_state']
        return result

    def __getitem__(self, item):
        return self.__dict__[item]


class MPFSBillingRepairer(object):

    def __init__(self, dry_run):
        self.dry_run = dry_run
        self.description = 'mpfs-admin-repair-billing.py ' + str(datetime.datetime.now())

    # ================= Modify methods ======================

    def _set_value(self, entity, key, prev_doc, new_val, msg, uid, sid, oid, dry_run=True):
        if not dry_run:
            entity.set(key, new_val)
            entity.set('description', self.description)
        log.info(format_log(msg=msg, uid=uid, sid=sid, oid=oid, prev_doc=prev_doc, now=new_val, key=key))

    def _create_a_subscription(self, service, oid, dry_run=True):
        mpfs_order = self._get_order(oid)
        mpfs_client = Client(service['uid'])
        mpfs_service = Service(service['_id'])
        if not dry_run:
            sub = Subscription.Create(mpfs_client, mpfs_order, mpfs_service)
            sub.set('description', self.description)
        log.info(format_log(msg='CHANGES:SUBSCRIPTION_CREATED', uid=service['uid'], sid=service['_id'], oid=oid))

    def _create_a_service(self, mpfs_order, bb_order, dry_run=True):
        mpfs_service = None
        client = Client(mpfs_order.uid)
        product = Product(mpfs_order.pid)
        if not dry_run:
            mpfs_service = billing.create_service_for_uid(client, product, mpfs_order, auto=bb_order.auto)
            mpfs_service.set('description', self.description)
            log.info(format_log(msg='CHANGES:SERVICE_CREATED', uid=bb_order.uid, sid=mpfs_service.sid,
                                oid=bb_order.oid))
        else:
            log.info(format_log(msg='CHANGES:SERVICE_CREATED', uid=bb_order.uid, sid=None, oid=bb_order.oid))
        return mpfs_service

    def _delete_subscription(self, subscription, dry_run=True):
        mpfs_subscription = Subscription(service=subscription['sid'], order=subscription['_id'])
        if not dry_run:
            mpfs_subscription.delete()
        log.info(format_log(msg='CHANGES:SUBSCRIPTION_DELETED', uid=subscription['uid'], sid=subscription['sid'],
                            oid=subscription['_id'], doc=subscription))

    def _give_fuckup_service(self, uid, new_amount, dry_run=True):
        fuckup_service = billing_services.find_one({'uid': uid, 'pid': COMMON_FUCKUP_PID})
        if fuckup_service:
            mpfs_fuckup_service = Service(fuckup_service['_id'])
            amount = int(mpfs_fuckup_service.product.attributes.amount)
            if not dry_run:
                mpfs_fuckup_service.set_attribute('product.amount', amount + new_amount)
            new_fuckup_service = billing_services.find_one({'_id': fuckup_service['_id']})
            log.info(format_log(msg='CHANGES:COMMON_FUCKUP_CHANGED', uid=fuckup_service['uid'],
                                sid=fuckup_service['_id'], prev=fuckup_service, now=new_fuckup_service))
        else:
            if not dry_run:
                mpfs_fuckup_service = billing_api.service_create(uid, COMMON_FUCKUP_PID, BONUS,
                                                                 attributes={'product.amount': new_amount})
                mpfs_fuckup_service.set('description', self.description)
                log.info(format_log(msg='CHANGES:COMMON_FUCKUP_CREATED', uid=uid, sid=mpfs_fuckup_service.sid,
                                    now=str(new_amount)))
            else:
                log.info(format_log(msg='CHANGES:COMMON_FUCKUP_CREATED', uid=uid, sid=None,
                                    now=str(new_amount)))

    def _fix_fuckup_services(self, uid, services, dry_run=True):
        # создаем один common_fuckup с объемом, равным сумме сервисов, затем удаляем эти сервисы
        new_amount = 0
        for i in services:
            new_amount += PRODUCT_DATA[i['pid']][ATTRIBUTES][AMOUNT]
        self._give_fuckup_service(uid, new_amount, dry_run)
        for i in services:
            if not dry_run:
                billing_api.service_delete(uid, i['_id'], i['pid'])
            log.info(format_log(msg='CHANGES:SERVICE_DELETED_TIME_IS_NONE', uid=uid, sid=i['_id'], pid=i['pid'], doc=i))
            subscription = billing_subscriptions.find_one({'sid': i['_id']})
            if not dry_run and subscription:
                self._delete_subscription(subscription, dry_run=dry_run)

    # =======================================================

    def _get_order(self, oid):
        try:
            order = ArchiveOrder(oid)
        except BillingOrderNotFound as e:
            order = Order(oid)
        return order

    def _find_oid_by_service(self, service):
        all_orders = list(billing_orders_history.find({'uid': service['uid'], 'sid': service['_id'],
                                                       'state': 'success'}))
        oid = None
        if all_orders:
            order = max(all_orders, key=lambda x: x.get('ctime'))
            oid = order.get('number', order.get('_id'))
        if not oid:
            order = billing_orders.find_one({'uid': service['uid'], 'sid': service['_id']})
            if order:
                oid = order.get('_id')
        if not oid and service['auto']:
            order = billing_subscriptions.find_one({'uid': service['uid'], 'sid': service['_id']})
            if order:
                oid = order.get('_id')
        # if not oid:
        #     # Ищем по совпадению uid и pid среди успешных заказов в некотором временном диапазоне.
        #     all_orders = list(billing_orders_history.find(
        #         {'uid': service['uid'], 'pid': service['pid'], 'state': 'success',
        #          'ctime': {'$gt': service['lbtime'] - TIME_SPAN, '$lt': service['lbtime'] + TIME_SPAN}}))
        #     if len(all_orders) == 1:
        #         order = all_orders[0]
        #         if 'sid' not in order or billing_services.find_one({'_id': order['sid']}) is None:
        #             oid = order.get('number', all_orders[0].get('_id'))
        #             log.info(format_log(msg='FOUND_OID_BY_PID_AND_TIME', uid=service['uid'], sid=service['_id'],
        #                                 pid=service['pid'], oid=oid))
        #             mpfs_order = self._get_order(oid)
        #             self._set_value(
        #                 mpfs_order, key='sid', prev_doc=order.get('sid'), new_val=service['_id'],
        #                 msg='CHANGES:ORDER_SID', uid=service['uid'], sid=service['_id'], oid=oid, dry_run=self.dry_run)
        #     elif len(all_orders) > 1:
        #         oids = list()
        #         for i in all_orders:
        #             oids.append(i.get('number', i.get('_id')))
        #         log.info(format_log(msg='SEVERAL_ORDERS_FOR_SERVICE', uid=service['uid'], sid=service['_id'],
        #                              pid=service['pid'], oid=oids))
        return oid

    def _find_services_by_bb_order(self, bb_order):
        services = list(billing_services.find({
            'uid': bb_order.uid,
            'pid': bb_order.pid,
            'lbtime': {'$gt': bb_order.lbtime - TIME_SPAN, '$lt': bb_order.lbtime + TIME_SPAN}
        }))
        sids = list()
        service = None
        if len(services) == 0:
            # Не нашли подходящий сервис
            log.info(format_log(msg='SERVICE_NOT_FOUND', uid=bb_order.uid, oid=bb_order.oid, pid=bb_order.pid,
                                ctime=bb_order.ctime, lbtime=bb_order.lbtime, btime=bb_order.btime))
        elif len(services) == 1:
            # Нашли сервис и он один, который нам подходит
            log.info(format_log(msg='FOUND_SERVICE_BY_PID_AND_TIME', uid=services[0]['uid'], sid=services[0]['_id'],
                                pid=services[0]['pid'], oid=bb_order.oid))
        else:
            # Несколько сервисов, которые подходят под условия
            for i in services:
                sids.append(i['_id'])
            log.info(format_log(msg='SEVERAL_SERVICES_BY_BB_ORDER', uid=bb_order.uid, oid=bb_order.oid,
                                 pid=bb_order.pid, sid=sids))
        return services

    def _get_bb_order(self, uid, oid, sid=None):
        bb_order = None
        try:
            try:
                bb_order = BB.check_order(uid, HOST, oid)
            except BigBillingBadResult, e:
                if (e.get_status_code() == 503 and
                     e.message == 'BigBillingBadResult: Order not found for service_order_id %s' % oid):
                    # BB не нашел заказа с таким oid
                    return None
                else:
                    raise
        except Exception as e:
            # Любая другая ошибка от BB
            error_log.error(format_log(msg='UNKNOWN_BB_ERROR', uid=uid, sid=sid, oid=oid), exc_info=True)
        return bb_order

    def _compare_times(self, service, bb_order, oid):
        # Все сравнения времен выполняем с погрешностью TIME_SPAN

        def compare_ts(time_key):
            greater_str = time_key.upper() + '_IS_GREATER'
            less_str = time_key.upper() + '_IS_LESS'
            changed_str = 'CHANGES:' + time_key.upper()
            delta = int(service[time_key]) - bb_order[time_key]
            if delta > TIME_SPAN:
                # bb.time < mpfs.time
                log.info(format_log(msg=greater_str, uid=service['uid'], sid=service['_id'], oid=oid,
                                    etime=bb_order[time_key], mpfs_time=service[time_key], auto=service['auto']))
            elif -delta > TIME_SPAN:
                # bb.time > mpfs.time ставим значение из ББ
                log.info(format_log(msg=less_str, uid=service['uid'], sid=service['_id'], oid=oid,
                                     mpfs_time=service[time_key], bb_time=bb_order[time_key], auto=service['auto']))
                mpfs_service = Service(service['_id'])
                self._set_value(
                    mpfs_service, key=time_key, prev_doc=service, new_val=bb_order[time_key],
                    msg=changed_str, uid=service['uid'], sid=service['_id'], oid=oid, dry_run=self.dry_run)
        compare_ts('btime')
        compare_ts('lbtime')

    def _check_base_services(self, uid, all_services):
        has_conflicts = False
        pids_set = {x['pid'] for x in all_services}
        # Выводим пользователей, у которых одновременно базовые услуг из двух групп (н-р, initial_3gb и initial_10gb)
        if (pids_set & init_services_old) and (pids_set & init_services_new):
            sids = [x['_id'] for x in all_services]
            pids = [x['pid'] for x in all_services]
            log.info(format_log(msg='EXTRA_BASE_SERVICES', uid=uid, sid=sids, pid=pids))
            has_conflicts = True
        # Выводим пользователей, у которых отсутствуют какие-либо базовые услуги
        elif not (pids_set & init_services_old) and not (pids_set & init_services_new):
            log.info(format_log(msg='NO_BASE_SERVICES', uid=uid))
            has_conflicts = True
        # Выводим пользователей у которого судя по всему должны быть старые базовые услуги, но некоторые из них
        # отсутствуют
        elif pids_set & init_services_old:
            for pid in init_services_old - pids_set:
                log.info(format_log(msg='MISSING_BASE_SERVICE', uid=uid, pid=pid))
                has_conflicts = True
        # Выводим дубликаты базовых услуг
        pid_checks = set()
        for service in filter(lambda x: x['pid'] in (init_services_old | init_services_new), all_services):
            if service['pid'] in pid_checks:
                log.info(format_log(msg='DUPLICATE_BASE_SERVICE', uid=uid, sid=service['_id'], pid=service['pid']))
                has_conflicts = True
            else:
                pid_checks.add(service['pid'])

        if has_conflicts:
            before_limit = Quota().limit(uid=uid)
            after_limit = repair_services(uid, dry_run=True)
            if after_limit < before_limit:
                # Результаты починки базовых сервисов уменьшат имеющийся объем у юзеров, выходим
                log.info(format_log(msg='REPAIRING_WILL_REDUCE_SPACE_OF_BASE_SERVICES', uid=uid,
                                     before=before_limit, after=after_limit))
                self._give_fuckup_service(uid, int(before_limit - after_limit), dry_run=self.dry_run)
                log.info(format_log(msg='SPACE_ENLARGED', uid=uid, before=before_limit, after=after_limit,
                                    delta=before_limit - after_limit))
            elif before_limit < after_limit:
                log.info(format_log(msg='REPAIRING_WILL_INCREASE_SPACE', uid=uid,
                                     before=before_limit, after=after_limit))
            repair_services(uid, dry_run=self.dry_run)
            log.info(format_log(msg='CHANGES:BASE_SERVICES', uid=uid))

    def _check_service_to_subscription(self, service, bb_order, oid):
        if bb_order.product_type == BB_TYPE_SUB and bb_order.subs_state in ACTIVE_SUB_BB_STATES:
            # По данным из ББ эта услуга является действующей подпиской
            if not service['auto']:
                # У сервиса auto == 0, хотя в ББ он подписка
                log.info(format_log(msg='NOT_AUTO_BUT_SUB', uid=service['uid'], sid=service['_id'], oid=oid,
                                     p_type=bb_order.product_type, state=bb_order.subs_state))
                mpfs_service = Service(service['_id'])
                self._set_value(mpfs_service, key='auto', prev_doc=service, new_val=True, msg='CHANGES:SERVICE_AUTO',
                                uid=service['uid'], sid=service['_id'], oid=oid, dry_run=self.dry_run)
            subscription = billing_subscriptions.find_one({'sid': service['_id']})
            if subscription is None:
                # Отсутствует подписка
                log.info(format_log(msg='MISSING_SUBSCRIPTION', uid=service['uid'], sid=service['_id'], oid=oid,
                                     state=service['state']))
                self._create_a_subscription(service, oid, dry_run=self.dry_run)
        else:
            # По данным из ББ эта услуга не является подпиской, либо ее 'subs_state' == 4. Подписки с таким
            # статусом в ББ не могут быть продлены! Поэтому, можем удалять подписку у нас, если надо.
            if service['auto']:
                log.info(format_log(msg='AUTO_BUT_NOT_SUBS', uid=service['uid'], sid=service['_id'], oid=oid,
                                     p_type=bb_order.product_type))
                mpfs_service = Service(service['_id'])
                self._set_value(mpfs_service, key='auto', prev_doc=service, new_val=False, msg='CHANGES:SERVICE_AUTO',
                                uid=service['uid'], sid=service['_id'], oid=oid, dry_run=self.dry_run)
            subscription = billing_subscriptions.find_one({'sid': service['_id']})
            if subscription is not None:
                # Подписка осталась существовать
                log.info(format_log(msg='UNDELETED_SUBSCRIPTION', uid=service['uid'], sid=service['_id'],
                                     oid=subscription['_id']))
                self._delete_subscription(subscription, dry_run=self.dry_run)

    def _check_primary_services(self, uid, all_services):
        fuckup_services = list()
        for service in all_services:
            if not service.get('btime'):
                # Если отсутствует btime, то должны создать common_fuckup. Для этого
                # запоминаем все такие услуги пользователя и затем выдаем один common_fuckup для них.
                log.info(format_log(msg='BTIME_IS_NONE', uid=uid, sid=service['_id']))
                fuckup_services.append(service)
                continue
            if not service.get('lbtime'):
                # Если отсутствует lbtime, то просто логируем
                log.info(format_log(msg='LBTIME_IS_NONE', uid=uid, sid=service['_id']))
                continue

            oid = self._find_oid_by_service(service)
            if oid is None:
                log.info(format_log(msg='MISSING_ORDER', uid=service['uid'], sid=service['_id'], pid=service['pid']))
                continue
            bb_order_dict = self._get_bb_order(uid, oid, service['_id'])
            if bb_order_dict is None:
                log.info(format_log(msg='NOT_FOUND_IN_BB', uid=uid, sid=service['_id'], oid=oid))
                continue

            # Проверяем, что у услуг в ББ присутствуют необходимые значения времен
            if BB_LBTIME not in bb_order_dict:
                log.info(format_log(msg='MISSING_BB_LBTIME', uid=uid, sid=service['_id'], oid=oid))
                continue
            if bb_order_dict['product_type'] == BB_TYPE_SUB and BB_BTIME not in bb_order_dict:
                log.info(format_log(msg='MISSING_BB_BTIME', uid=uid, sid=service['_id'], oid=oid))
                continue

            bb_order = BBOrderInfo.create_from_bb_order(bb_order_dict, oid, uid)
            self._check_service_to_subscription(service, bb_order, oid)
            self._compare_times(service, bb_order, oid)

        if fuckup_services:
            self._fix_fuckup_services(uid, fuckup_services, dry_run=self.dry_run)

    def _check_services(self, uid):
        # Разбираемся с базовыми сервисами
        base_pids = list(init_services_old) + list(init_services_new)
        base_services = list(billing_services.find({'uid': uid, 'pid': {'$in': base_pids}}))
        self._check_base_services(uid, base_services)

        # Разбираемся с покупными сервисами
        primary_services = list(billing_services.find({'uid': uid, 'pid': {'$in': PRODUCT_IDS}}))
        self._check_primary_services(uid, primary_services)

    def _check_subscription_consistency(self, uid):
        for subscription in billing_subscriptions.find({'uid': uid}):
            service = billing_services.find_one({'_id': subscription['sid']})
            if service is not None:
                continue
            # Подписка, у которой отсутствует Service.
            log.info(format_log(msg='MISSING_SERVICE', uid=uid, sid=subscription['sid'], oid=subscription['_id']))
            bb_order_dict = self._get_bb_order(uid, subscription['_id'], subscription['sid'])
            if bb_order_dict is None:
                log.info(format_log(msg='NOT_FOUND_IN_BB', uid=uid, sid=subscription['sid'], oid=subscription['_id']))
                continue
            if BB_LBTIME not in bb_order_dict:
                # Такие сервисы выданы по ошибке.
                log.info(format_log(msg='MISSING_BB_LBTIME', uid=uid, sid=subscription['sid'], oid=subscription['_id'],
                                    bb_status=bb_order_dict.get('status'),
                                    bb_sub_state=bb_order_dict.get('subs_state')))
                if bb_order_dict.get('subs_state') not in ACTIVE_SUB_BB_STATES:
                    self._delete_subscription(subscription, dry_run=self.dry_run)
                continue
            bb_order = BBOrderInfo.create_from_bb_order(bb_order_dict, subscription['_id'], uid)
            if (bb_order.btime < ctimestamp() + TIME_SPAN or
                bb_order.product_type == BB_TYPE_SUB and bb_order.subs_state not in ACTIVE_SUB_BB_STATES):
                # Эта запись неактивна в ББ, удаляем подписку
                log.info(format_log(msg='UNDELETED_SUBSCRIPTION', uid=uid, sid=subscription['sid'],
                                    oid=subscription['_id']))
                self._delete_subscription(subscription, dry_run=self.dry_run)
                continue

            try:
                mpfs_order = self._get_order(bb_order.oid)
            except BillingOrderNotFound:
                log.info(format_log(msg='NO_ORDER_FOR_SUBSCRIPTION', uid=uid, sid=subscription['_id'],
                                     oid=subscription['sid']))
                continue
            log.info(format_log(msg='MISSING_ACTIVE_SERVICE', uid=uid, sid=subscription['sid'],
                                oid=subscription['_id']))
            # Создаем сервис для подписки при необходимости и переставляем сиды
            # mpfs_subscription = Subscription(order=subscription['_id'])
            # services = self._find_services_by_bb_order(bb_order)
            # if len(services) == 0:
            #     mpfs_service = self._create_a_service(mpfs_order, bb_order, dry_run=self.dry_run)
            # elif len(services) == 1:
            #     mpfs_service = Service(services[0]['_id'])
            # else:
            #     continue
            # if self.dry_run:
            #     self._set_value(
            #         mpfs_order, key='sid', prev_doc=subscription, new_val=None, msg='CHANGES:ORDER_SID',
            #         uid=subscription['uid'], sid=None, oid=subscription['_id'], dry_run=self.dry_run)
            #     self._set_value(
            #         mpfs_subscription, key='sid', prev_doc=subscription, new_val=None, msg='CHANGES:SUBSCRIPTION_SID',
            #         uid=subscription['uid'], sid=None, oid=subscription['_id'], dry_run=self.dry_run)
            # else:
            #     self._set_value(
            #         mpfs_order, key='sid', prev_doc=subscription, new_val=mpfs_service.sid, msg='CHANGES:ORDER_SID',
            #         uid=subscription['uid'], sid=mpfs_service.sid, oid=subscription['_id'], dry_run=self.dry_run)
            #     self._set_value(
            #         mpfs_subscription, key='sid', prev_doc=subscription, new_val=mpfs_service.sid,
            #         msg='CHANGES:SUBSCRIPTION_SID', uid=subscription['uid'], sid=mpfs_service.sid,
            #         oid=subscription['_id'], dry_run=self.dry_run)

    def check_mpfs_billing(self, uid):
        """
        Проверить консистентность биллинговых данных в базе MFPS для данного юзера

        :param uid: uid
        """
        try:
            user = user_index.check_user(uid)
            # Нас не интересуют юзеры, которые не являются стандартными
            if not user or 'type' not in user or user['type'] != 'standart':
                user_type = user.get('type') if user else None
                log.info(format_log(msg='SKIP_USER', uid=uid, type=user_type))
                return
            self._check_services(uid)
            self._check_subscription_consistency(uid)
        except Exception as e:
            error_log.error(format_log(msg='UNEXPECTED_EXCEPTION', uid=uid), exc_info=True)

    def check_from_bb(self, bb_record):
        """
        Проверить консистентность биллинговых данных заказа, выгруженного из базы ББ.

        :param bb_record: информация о заказе в ББ (формат смотреть в BBOrderInfo)
        """
        bb_order = BBOrderInfo(bb_record)
        # Получаем актуальную информацию о заказе, так как данные в файле могли устареть
        bb_order_dict = self._get_bb_order(bb_order.uid, bb_order.oid)
        if bb_order_dict is None:
            log.info(format_log(msg='NOT_FOUND_IN_BB', uid=bb_order.uid, sid=None, oid=bb_order.oid))
            return
        if BB_LBTIME not in bb_order_dict:
            # Такие сервисы выданы по ошибке.
            log.info(format_log(msg='MISSING_BB_LBTIME', uid=bb_order.uid, sid=None, oid=bb_order.oid,
                                bb_status=bb_order_dict.get('status'),
                                bb_sub_state=bb_order_dict.get('subs_state')))
            return
        bb_order = BBOrderInfo.create_from_bb_order(bb_order_dict, bb_order.oid, bb_order.uid)
        if (bb_order.btime < ctimestamp() + TIME_SPAN or
            bb_order.product_type == BB_TYPE_SUB and bb_order.subs_state not in ACTIVE_SUB_BB_STATES):
            # Эта услуга уже истекла, для нее ничего не ищем
            return

        # Ищем Order в MPFS базе
        order = billing_orders_history.find_one({'number': bb_order.oid, 'uid': bb_order.uid})
        if not order:
            order = billing_orders_history.find_one({'_id': bb_order.oid})
        if not order:
            order = billing_orders.find_one({'_id': bb_order.oid})
        if not order:
            # Не нашли Order; Пока таких ошибок не было
            log.info(format_log(msg='ORDER_NOT_FOUND', uid=bb_order.uid, oid=bb_order.oid, pid=bb_order.pid,
                                 ctime=bb_order.ctime, lbtime=bb_order.lbtime))
            return
        mpfs_order = self._get_order(bb_order.oid)

        # Ищем Service в MPFS, который соответствует сервису из ББ
        service = None

        if 'sid' in order:
            service = billing_services.find_one({'_id': order.get('sid')})
        if service is None:
            # Ищем заказ по обратной ссылке
            service = billing_services.find_one({'order': bb_order.oid, 'uid': bb_order.uid})
        # if service is None:
        #     # Ищем по совпадающим пидам, юзеру и времени последней оплаты
        #     services = self._find_services_by_bb_order(bb_order)
        #     if len(services) == 0:
        #         mpfs_service = self._create_a_service(mpfs_order, bb_order, dry_run=self.dry_run)
        #         service = billing_services.find_one({'_id': mpfs_service.sid})
        #     elif len(services) == 1:
        #         service = services[0]
        if service is None:
            log.info(format_log(msg='SERVICE_NOT_FOUND', uid=bb_order.uid, oid=bb_order.oid))
            return

        # Связываем заказ с найденным сервисом
        if 'sid' not in order:
            log.info(format_log(msg='ORDER_HAS_NO_SID', uid=bb_order.uid, oid=bb_order.oid))
            self._set_value(
                mpfs_order, key='sid', prev_doc=order, new_val=service['_id'], msg='CHANGES:ORDER_SID_CREATED',
                uid=service['uid'], sid=service['_id'], oid=bb_order.oid, dry_run=self.dry_run)
        elif order['sid'] != service['_id']:
            log.info(format_log(msg='SERVICE_WITH_ORDER_SID_DOESNT_EXIST', uid=bb_order.uid, oid=bb_order.oid,
                                sid=order.get('sid')))
            self._set_value(
                mpfs_order, key='sid', prev_doc=order, new_val=service['_id'], msg='CHANGES:ORDER_SID',
                uid=service['uid'], sid=service['_id'], oid=bb_order.oid, dry_run=self.dry_run)

        subscription = billing_subscriptions.find_one({'_id': 'bb_order.oid'})
        if subscription is not None and subscription['sid'] != service['_id']:
            log.info(format_log(msg='SIDS_DOESNT_MATCH', uid=bb_order.uid, oid=bb_order.oid,
                                ser_sid=order.get('sid'), sub_sid=subscription['sid']))
            sub_service = billing_services.find_one({'_id': subscription['sid']})
            if sub_service is not None:
                log.info(format_log(msg='SERVICE_FOR_SUBSCRIPTION_EXISTS', uid=bb_order.uid, oid=bb_order.oid,
                                    sid=subscription['sid']))
                return
            log.info(format_log(msg='SERVICE_WITH_SUBSCRIPTION_SID_DOESNT_EXIST', uid=bb_order.uid, oid=bb_order.oid,
                                sid=subscription['sid']))
            mpfs_subscription = Subscription(service=subscription['sid'], order=subscription['_id'])
            self._set_value(
                mpfs_subscription, key='sid', prev_doc=subscription, new_val=service['_id'],
                msg='CHANGES:SUBSCRIPTION_SID', uid=service['uid'], sid=service['_id'], oid=bb_order.oid,
                dry_run=self.dry_run)

        # Чиним консистентность с подписками для этого сервиса
        self._check_service_to_subscription(service, bb_order, bb_order['oid'])
        self._compare_times(service, bb_order, bb_order['oid'])


option_list = (
    optparse.Option('-f', '--file',
                    type='string', action='store',
                    dest='uid_file', default=None,
                    help='File where to store local uid cache. (Optional parameter)'),
    optparse.Option('-b', '--bb_file',
                    type='string', action='store',
                    dest='bb_file', default=None,
                    help='File where to store active services from BB. (Optional parameter)'),
    optparse.Option('-p', '--processes',
                    type='int', action='store',
                    dest='processes', default=None,
                    help='Number of processes. Default is your CPU count: %d.' % multiprocessing.cpu_count()),
    optparse.Option('-d', '--dry_run', action='store_true', dest='dry_run', default=False,
                    help='Do not repair anything'),
)

if __name__ == '__main__':
    parser = optparse.OptionParser(option_list=option_list)
    (options, args) = parser.parse_args()
    repairer = MPFSBillingRepairer(options.dry_run)
    # Чиним по uid'ам
    if options.uid_file:
        iterdbuids.run(repairer.check_mpfs_billing, None, options.uid_file, max_processes=options.processes)
    # Чиним по данным из биллинга
    if options.bb_file:
        with open(options.bb_file) as f:
            for bb_record in f:
                try:
                    repairer.check_from_bb(bb_record)
                except Exception as e:
                    error_log.error(format_log(msg='UNEXPECTED_EXCEPTION', bb_record=bb_record), exc_info=True)
