#!/usr/bin/python
# -*- coding: utf-8 -*-

import time
import sys
import os
import traceback


if os.getuid() != 0:
    print "you must be root to run this script"
    sys.exit(1)

from optparse import OptionParser, Option
from datetime import datetime
from dateutil.relativedelta import relativedelta
from collections import Counter

import mpfs.engine.process
mpfs.engine.process.setup_admin_script()
from mpfs.core.billing.interface.services import service_set_attribute
from mpfs.core.billing import Service


db = mpfs.engine.process.dbctl().database()
log = mpfs.engine.process.get_error_log()


def update_btime(sid, btime_ts, dry_run=True):
    service = Service(sid)
    if service:
        if not dry_run:
            service.set_attribute('service.btime', btime_ts)
    else:
        raise Exception('Service not found')


def prolong(services, dry_run=True):
    current_dt = datetime.now()
    counter = Counter(total=0, skipped=0, need_to_prolong=0, success=0, error=0)

    for service in services:
        btime = service.get('btime')
        uid = service['uid']
        sid = service['_id']

        ctime_ts = int(service['ctime'])
        ctime_dt = datetime.fromtimestamp(ctime_ts)

        new_btime_dt = None
        if btime:
            # btime есть
            btime_ts = int(btime)
            btime_dt = datetime.fromtimestamp(btime_ts)
            if btime_dt < current_dt + relativedelta(months=+2):
                new_btime_dt = btime_dt + relativedelta(months=+3)
        else:
            # нет btime - дату предоставления услуги "переносим" на текущий месяц
            # и прибавляем 3 месяца
            new_btime_dt = ctime_dt + relativedelta(months=relativedelta(current_dt, ctime_dt).months + 1 + 3)

        counter['total'] += 1
        if new_btime_dt is not None:
            # меняем услугу
            counter['need_to_prolong'] += 1
            new_btime_ts = int(time.mktime(new_btime_dt.timetuple()))
            try:
                update_btime(sid, new_btime_ts, dry_run)
                counter['success'] += 1
                log.info('Prolongation success. uid: "%s", sid: "%s", new_btime: "%s"' % (uid, sid, str(new_btime_ts)))
            except Exception:
                counter['error'] += 1
                log.info('Prolongation error. uid: "%s", sid: "%s", error: %s' % (uid, sid, str(traceback.format_exc())))
        else:
            # не меняем услугу
            log.info('Prolongation skip. uid: "%s", sid: "%s", new_btime: "%s"' % (uid, sid, str(btime)))
            counter['skipped'] += 1
    return counter


usage = """
usage:
%prog --uid 123
%prog --all
"""

option_list = (
    Option(
        '-u',
        '--uid',
        type='string',
        action='store',
        dest='uid',
        help='run script for one user'
    ),
    Option(
        '--all',
        action='store_true',
        dest='all',
        default=False,
        help='run script for all users'
    ),
    Option(
        '--dry-run',
        action='store_true',
        dest='dry_run',
        default=False,
        help="run without commit",
    ),
)

parser = OptionParser(usage, option_list=option_list)
(options, args) = parser.parse_args()


def get_rostelecom_services(uid=None):
    query = {"pid": "rostelecom_2014_100gb"}
    if uid is not None:
        query['uid'] = uid
    return list(db['billing_services'].find(query))


if __name__ == "__main__":
    if options.uid:
        services = get_rostelecom_services(options.uid)
    elif options.all:
        services = get_rostelecom_services()
    else:
        print 'Need option "--uid <UID>" or "--all"'
        sys.exit(1)

    if services:
        log.info("Found '%i' rostelecom services" % len(services))
        counter = prolong(services, dry_run=options.dry_run)
        print 'Total: "{total}", Need_to_prolong: "{need_to_prolong}"'\
              ' (success: "{success}", error: "{error}"), Skipped: "{skipped}"'.format(**counter)
        print "More info: 'cat /var/log/mpfs/admin.log | grep -F \'INFO mpfs-admin-rostelecom-prolong\''"
    else:
        print 'Services not found'
