#!/usr/bin/python
# -*- coding: utf-8 -*-
import os

os.environ['MPFS_PACKAGE'] = 'disk'
import traceback

from optparse import OptionParser, Option
from pymongo.errors import ConnectionFailure

import mpfs.engine.process
from mpfs.common.util import iterdbuids

mpfs.engine.process.setup_admin_script()
admin_log = mpfs.engine.process.get_default_log()

from mpfs.config import settings
from mpfs.common.util.database.connections import DirectRSConnectionFabric
from mpfs.core.filesystem.cleaner.models import DeletedStid, DeletedStidSources
from mpfs.core.filesystem.resources.base import Resource
from mpfs.metastorage.mongo.collections.filesystem import UserCollectionZipped

DATA_COLLECTIONS = settings.mongo['other']['file_collections']

connection_fabric = DirectRSConnectionFabric()
user_data_collection = UserCollectionZipped()
fsync_safe_w = mpfs.engine.process.dbctl().fsync_safe_w()

option_list = (
    Option(
        '-f', '--file',
        type='string', action='store',
        dest='file', help='file with uids',
    ),
    Option(
        '-u', '--uid',
        type='string', action='store',
        dest='uid', help='UID'
    ),
    Option(
        '--dry-run',
        action='store_true',
        dest='dry_run', help='No modifications',
        default=False,
    ),
)

usage = "usage: %prog -f UIDS"
parser = OptionParser(usage, option_list=option_list)
(options, args) = parser.parse_args()


def main(uid):
    limit = 1000
    unique_file_ids = []

    for collection_name in DATA_COLLECTIONS:
        data_collection = connection_fabric.get_collection_for_uid(
            collection_name, uid)
        uid_name = connection_fabric._get_collection_uid_name(collection_name)
        spec = {
            uid_name: uid,
            'hid': {'$exists': 1},
            '$or': [
                {'data.stids': {'$exists': 0}},
                {'data.file_id': {'$exists': 0}}
            ]
        }

        processed = 0
        to_process = data_collection.find(spec).count()
        found = list(data_collection.find(spec, limit=limit))
        while processed < to_process and found:
            for element in found:
                processed += 1
                if not ('data' in element and 'zdata' in element):
                    continue
                try:
                    rawdata = user_data_collection._unzip_resource_data(
                        element['type'], element['data'], element['zdata'])

                    # fix file_ids
                    file_id = rawdata.get('meta', {}).get('file_id')
                    if not file_id or file_id is None:
                        file_id = Resource.generate_file_id(uid, element['key'])
                        unique_file_ids.append(file_id)

                    else:
                        if file_id in unique_file_ids:
                            file_id = Resource.generate_file_id(uid, element['key'])
                        unique_file_ids.append(file_id)

                    # remove old previews
                    preview_mids_to_delete = []
                    if 'previews' in rawdata.get('meta', {}):
                        for _, mid in rawdata['meta']['previews'].iteritems():
                            preview_mids_to_delete.append(mid)
                    zdata, data = user_data_collection._zip_file_data(rawdata)

                    if preview_mids_to_delete:
                        for each in data['stids']:
                            if each['type'] == 'pmid' and each['stid'] in preview_mids_to_delete:
                                preview_mids_to_delete.remove(each['stid'])
                        try:
                            DeletedStid.controller.bulk_create([
                                DeletedStid(stid=stid, stid_source=DeletedStidSources.ADMIN_UNZIP_MIDS)
                                for stid in preview_mids_to_delete
                            ])
                        except Exception:
                            admin_log.error('Failed to put preview mids to delete_stids')

                    # update
                    update_spec = {
                        '_id': element['_id'],
                        'uid': uid,
                        'version': element['version'],
                    }
                    update_doc = {
                        '$set': {
                            'zdata': zdata,
                            'data.stids': data['stids'],
                            'data.file_id': file_id,
                        }
                    }

                    admin_log.debug('Updating %s:%s' % (uid, element['key']))
                    if not options.dry_run:
                        fails_count = 0
                        while fails_count < 3:
                            try:
                                data_collection.update(update_spec, update_doc, **fsync_safe_w)
                            except ConnectionFailure:
                                fails_count += 1
                            else:
                                break

                except Exception:
                    admin_log.error('ERROR: element %s' % element)
                    admin_log.error(traceback.format_exc())

            if options.dry_run:
                if processed >= to_process:
                    break
                else:
                    found = list(data_collection.find(spec, skip=processed, limit=limit))
            else:
                found = list(data_collection.find(spec, limit=limit))


def callback(result):
    admin_log.info(result)


if __name__ == '__main__':
    if options.file or options.uid:
        if options.file:
            iterdbuids.run(main, callback, options.file)
        else:
            main(options.uid)
