#!/usr/bin/python
# -*- coding: utf-8 -*-
"""
Скрипт пересчёта хэшей файлов по UID'у

https://st.yandex-team.ru/CHEMODAN-24149
"""
import os
import sys
import time
import signal

from optparse import OptionParser, Option
from collections import defaultdict
from itertools import product
from multiprocessing import Process
from contextlib import contextmanager

import mpfs.engine.process

mpfs.engine.process.setup_admin_script()

from mpfs.core.operations import manager
from mpfs.common.errors import StorageInitUser
from mpfs.core.user.base import User
from mpfs.metastorage.mongo.collections.filesystem import UserDataCollection
from mpfs.common.util.logger import TSKVMessage
from mpfs.common.util import format_dict_table


log = mpfs.engine.process.get_default_log()
usage = "usage: sudo -u nginx /usr/sbin/%prog -h"

option_list = (
    Option(
        '-u', '--uid',
        action='store',
        dest='uid',
        type='string',
        help='user uid'
    ),
    Option(
        '-p', '--path',
        action='store',
        dest='path',
        type='string',
        help='user path'
    ),
    Option(
        '-v', '--verbose',
        action='store_true',
        dest='verbose',
        help='verbose everything (False by default)',
        default=False,
    ),
    Option(
        '--stat',
        action='store_true',
        dest='stat',
        help='get statistic for running operations',
        default=False,
    ),
)

parser = OptionParser(usage, option_list=option_list)
(options, args) = parser.parse_args()


def say(something):
    log.info(something)
    if options.verbose:
        print something


@contextmanager
def timeout(timeout=None):
    """
    При входе в контекст менеджер создается процесс,
    который через таймаут убивает процесс-родитель и завершается
    """
    def process_terminator(pid, timeout=None):
        if timeout is not None:
            time.sleep(timeout)
        os.kill(pid, signal.SIGKILL)

    try:
        self_pid = os.getpid()
        terminator = Process(target=process_terminator, args=(self_pid, timeout))
        terminator.start()
        yield
    finally:
        os.kill(terminator.pid, signal.SIGKILL)


def _worker(item):
    # игнорируем воркером сигналы
    for sig in (signal.SIGTERM, signal.SIGINT):
        signal.signal(sig, lambda x, y: None)
    # работаем не более 5 секунд
    with timeout(timeout=5):
        unpacked_item = UserDataCollection().unpack_single_element(item)
        uid = unpacked_item['uid']
        # ставим операции пересчета хеш-ей
        operation = manager.create_operation(uid, 'system', 'update_file_hash', odata={'path': unpacked_item['key']})
        result = {
            'utime': unpacked_item['data'].get('utime', None),
            'mtime': unpacked_item['data'].get('mtime', None),
            'ctime': unpacked_item['data'].get('ctime', None),
            'mimetype': unpacked_item['data'].get('mimetype', None),
            'path': unpacked_item['key'],
            'oid': operation.id,
            'uid': uid,
        }
        say(TSKVMessage(**result))


def update_file_hash(uid, path=None):
    try:
        User(uid)
    except StorageInitUser:
        say('user %s is not exist' % uid)
        sys.exit(1)

    if not mpfs.engine.process.usrctl().is_writable(uid):
        say('user %s is in readonly state' % uid)
        sys.exit(1)

    start_time = time.time()
    spec = {'uid': uid, 'type': 'file'}
    if path is not None:
        spec['key'] = path

    parent_pid = os.getpid()
    say('Starting update for %s' % uid)

    # на сигналы убиваем только parenta
    for sig in (signal.SIGTERM, signal.SIGINT):
        signal.signal(sig, lambda x, y: exit(1) if parent_pid == parent_pid else None)

    for item in list(UserDataCollection().find_all(spec)):
        p = Process(target=_worker, args=(item,))
        p.start()
        p.join()
        if p.exitcode == -9:
            say('Job stopped by timeout')
            exit(1)
        elif p.exitcode == 1:
            exit(1)

    end_time = time.time()
    say('Finished update for %s' % uid)
    say('TOTAL ELAPSED: %.3f sec' % (end_time - start_time))


def get_stat(uids):
    count = defaultdict(lambda: defaultdict(int))
    uniq_uids = {'_total_'}
    uniq_ops = {'_total_'}
    for uid in uids:
        uniq_uids.add(uid)
        for op in manager.get_active_operations(uid):
            op_name = "%s.%s" % (op.get('type', '-'), op.get('subtype', '-'))
            uniq_ops.add(op_name)
            count[uid][op_name] += 1
            count[uid]['_total_'] += 1
            count['_total_'][op_name] += 1
            count['_total_']['_total_'] += 1
    result = []
    for uid, op in product(uniq_uids, uniq_ops):
        result.append({'uid': uid, 'operation': op, 'count': count[uid][op]})
    print format_dict_table(result, ['operation', 'uid', 'count'])


def stdin_stat():
    return get_stat({l.strip('\n ') for l in sys.stdin})


if __name__ == "__main__":
    if options.stat:
        print ('OPERATION STAT MODE')
        stdin_stat()
    elif options.uid and options.path:
        say('UPDATE HASHES MODE')
        update_file_hash(options.uid, path=options.path)
    else:
        parser.print_help()
        sys.exit(0)
