#!/usr/bin/python
# -*- coding: utf-8 -*-
import os
os.environ['MPFS_PACKAGE'] = 'disk'
import traceback

from optparse import OptionParser, Option
from pymongo.errors import ConnectionFailure

import mpfs.engine.process

from mpfs.common.util import logger
mpfs.engine.process.setup_admin_script()
log = mpfs.engine.process.get_default_log()

from mpfs.config import settings
from mpfs.common.util import iterdbuids
from mpfs.common.util import filetypes
from mpfs.core import factory
from mpfs.core.address import Address
from mpfs.common.util.database.connections import DirectRSConnectionFabric
from mpfs.metastorage.mongo.collections.filesystem import UserCollectionZipped

DATA_COLLECTIONS = settings.mongo['other']['file_collections']

connection_fabric = DirectRSConnectionFabric()
user_data_collection = UserCollectionZipped()
fsync_safe_w = mpfs.engine.process.dbctl().fsync_safe_w()

option_list = (
    Option(
        '-f', '--file',
        type='string', action='store',
        dest='file', help='file with uids',
    ),
    Option(
        '-u', '--uid',
        type='string', action='store',
        dest='uid', help='UID'
    ),
    Option(
        '-t', '--type',
        type='string', action='store',
        dest='type', help='mt from filetypes.builtin_extensions',
        default=None,
    ),
    Option(
        '--dry-run',
        action='store_true',
        dest='dry_run', help='No modifications',
        default=False,
    ),
)

usage = "usage: %prog -f UIDS"
parser = OptionParser(usage, option_list=option_list)
(options, args) = parser.parse_args()


def main(uid):
    limit = 1000
    for collection_name in DATA_COLLECTIONS:
        data_collection = connection_fabric.get_collection_for_uid(collection_name, uid)
        uid_name = connection_fabric._get_collection_uid_name(collection_name)
        
        spec = {
            uid_name: uid,
            'data.mt': filetypes.getGroupNumber(options.type),
        }
        
        processed = 0
        to_process = data_collection.find(spec).count()
        found = list(data_collection.find(spec, limit=limit))
        
        while processed < to_process and found:
            for element in found:
                processed += 1
                
                if element['type'] == 'dir':
                    continue
                
                
                try:
                    resource = factory.get_resource(uid, Address.Make(uid, element['key']))
                    new_media_type = filetypes.getGroupByName(resource.name, resource.mimetype)
                    new_mt_integer = filetypes.getGroupNumber(new_media_type)
                    
                    if new_mt_integer == element['data']['mt']:
                        continue
                    
                    update_spec = {
                        '_id': element['_id'],
                        'uid': uid,
                        'version': element['version'],
                    }
                    update_doc = {
                        '$set': {'data.mt': new_mt_integer}
                    }
                    
                    if not options.dry_run:
                        fails_count = 0
                        while fails_count < 3:
                            try:
                                data_collection.update(
                                    update_spec, update_doc, **fsync_safe_w()
                                )
                            except ConnectionFailure:
                                fails_count += 1
                            else:
                                break
        
                    log.info(
                        'Updated %s:%s to %s' % \
                        (uid, element['key'], new_media_type)
                    )
        
                except Exception:
                    log.error('ERROR: element %s' % element)
                    log.error(traceback.format_exc())
                    
            if options.dry_run:
                if processed >= to_process:
                    break
                else:
                    found = list(data_collection.find(spec, skip=processed, limit=limit))
            else:
                found = list(data_collection.find(spec, limit=limit))
   


def callback(result):
    log.info(result)


if __name__ == '__main__':
    if options.file or options.uid:
        if options.file:
            iterdbuids.run(main, callback, options.file)
        else:
            main(options.uid)
