#!/usr/bin/python
# -*- coding: utf-8 -*-
"""

Скрипт, коннектящий на нужный шард
Запускается, находит данные для покдлючения и передает управление консоли монги или postgres

"""
import operator
import random
import re
import subprocess
import sys
from argparse import ArgumentParser

from pymongo import MongoClient, MongoReplicaSetClient
import mpfs.engine.process
from mpfs.metastorage.postgres.query_executer import PGQueryExecuter


class MongoConnect(object):
    def __init__(self, target, use_primary, use_ipv4):
        self.target = target
        self.use_primary = use_primary
        self.use_ipv4 = use_ipv4

    def get_connection(self):
        from mpfs.config import settings
        if self.target in settings.mongo["connections"]:
            return mpfs.engine.process.dbctl().connection(self.target)
        return mpfs.engine.process.dbctl().mapper.rspool.get_connection_for_rs_name(self.target)

    def get_host_port(self, conn):
        from mpfs.metastorage.mongo.util import LazyDatabaseObject
        if isinstance(conn, LazyDatabaseObject):
            # Force Lazy connection
            conn = conn.get_instance()

        if isinstance(conn, MongoClient):
            hosts = conn.nodes
            primary = conn.host, conn.port
        elif isinstance(conn, MongoReplicaSetClient):
            hosts = conn.hosts
            primary = conn.primary
        else:
            raise NotImplemented('Unknown connection instance type')

        replicas = filter(lambda x: x != primary, hosts)
        if self.use_primary or not replicas:
            return primary
        else:
            return random.choice(replicas)

    def get_cmdline(self, host, port):
        command = ['mongo', '--host', host, '--port', str(port)]
        if not self.use_primary:
            command.extend(['--eval', "'db.getMongo().setSlaveOk()'", '--shell'])
        if not self.use_ipv4:
            command.append('--ipv6')
        return " ".join(command)

    def connect(self):
        conn = self.get_connection()
        host, port = self.get_host_port(conn)
        command = self.get_cmdline(host, port)
        print "Executing: " + command
        subprocess.call(command, shell=True)


class PostgresConnect(object):
    def __init__(self, target, use_primary):
        self.target = target
        self.use_primary = use_primary
        self.pg_query_executer = PGQueryExecuter()

    def get_shard(self):
        try:
            int(target)
        except ValueError:
            func = self.pg_query_executer._sharpei.get_shard_by_name
        else:
            func = self.pg_query_executer._get_shard_by_shard_id
        return func(target)

    def get_shardhost(self, shard):
        if self.use_primary:
            return shard.get_master()
        else:
            return shard.get_random_slave()

    def get_cmdline(self, shardhost):
        command = ['psql', shardhost.get_connection_string()]
        return " ".join(command)

    def connect(self):
        shard = self.get_shard()
        shardhost = self.get_shardhost(shard)
        command = self.get_cmdline(shardhost)

        password_match = re.match('.*postgresql://[^:]*:([^@]*).*', command)
        print "Executing: " + command.replace(password_match.group(1), "*" * len(password_match.group(1)))

        subprocess.call(command, shell=True)


def connect(target, use_primary, use_ipv4):
    try:
        PostgresConnect(target, use_primary).connect()
    except Exception:
        MongoConnect(target, use_primary, use_ipv4).connect()


def show_all():
    from mpfs.common.util import format_dict_table
    from mpfs.config import settings

    connections = sorted(settings.mongo["connections"].values(), key=operator.itemgetter('name'))
    print "Config connections:"
    print format_dict_table(connections, ('name', 'replicaSet', 'host'))
    print "Postgres:"
    pg_shards_pretty = sorted([
        {'id': i._id, 'name': i._name, 'hosts': ",".join("%s" % h for h in i._hosts)}
        for i in PGQueryExecuter._sharpei.get_all_shards() + [PGQueryExecuter._sharpei.get_common_shard()]
    ], key=operator.itemgetter('name'))
    print format_dict_table(pg_shards_pretty, ('id', 'name', 'hosts'))


if __name__ == "__main__":
    parser = ArgumentParser(description='Connect to MPFS database')
    parser.add_argument('-u', '--uid', dest='uid', help='user uid')
    parser.add_argument('-c', '--connection', dest='connection', help='connection string from config to connect to')
    parser.add_argument('-s', '--shard', dest='shard', help='shard name to connect to. Example: disk-unit-01')
    parser.add_argument('-l', '--list', dest='list', action='store_true', help='list all shards/connection strings')
    parser.add_argument('--primary', dest='primary', action='store_true', help='connect to primary (default: False)')
    parser.add_argument('--ipv4', dest='ipv4', action='store_true', default=False,
                        help='force use IPv4. (default: IPv6)')
    args = parser.parse_args()
    if args.list:
        mpfs.engine.process.setup_anyone_script()
        show_all()
    elif args.connection or args.shard or args.uid:
        mpfs.engine.process.setup_anyone_script()
        if args.uid:
            pg_query_executer = PGQueryExecuter()
            if pg_query_executer.is_user_in_postgres(args.uid):
                target = pg_query_executer.get_shard_id(args.uid)
            else:
                target = mpfs.engine.process.usrctl().info(args.uid)['shard']
        else:
            target = args.shard or args.connection
        connect(target, args.primary, args.ipv4)
    else:
        parser.print_help()
        sys.exit(2)
