#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Выводим информацию по тем сервисам, которые есть ББ, но отсутствуют в MPFS

На STDIN oid,type,pid,uid,ctime_msec,btime_msec,lbtime_msec

В STDOUT msg=<имя конфликта>\tuid=<uid>\toid=<order_number>\tpid=<product_name>\tctime=<service_ctime>\t
         lbtime=<service_lbtime>\tbtime=<service_btime>
"""
import logging
import datetime
import multiprocessing
import optparse
import sys

from dateutil.relativedelta import relativedelta
from mpfs.engine.process import setup_anyone_script

setup_anyone_script()

from mpfs.core.metastorage.control import (
    billing_orders,
    billing_services,
    billing_orders_history,
    billing_services_history,
)
from mpfs.common.static.tags.billing import (
    MONTH,
    PERIOD
)
from mpfs.common.util import ctimestamp
from mpfs.core.billing.product.catalog import PRIMARY_PRODUCTS
log = logging.getLogger('billing_log')
log.propagate = False
billing_checks_handler = logging.FileHandler('mpfs-find-missing-services.log')
log.addHandler(billing_checks_handler)
log.setLevel(logging.INFO)

TIME_SPAN = 1800
PRODUCT_DATA = {x['id']: x for x in PRIMARY_PRODUCTS}


class BBOrderInfo(object):

    def __init__(self, bb_record):
        # парсинг
        args = bb_record.strip().split(',')
        bb_pid = args[2][1:-1]
        ptype = args[1][1:-1]
        # вытаскиваем pid как в mpfs
        self.pid = bb_pid[:bb_pid.rfind(ptype) - 1]
        self.ctime = args[4][:-3]
        self.lbtime = args[6][:-3]
        if args[5]:
            self.btime = args[5][:-3]
        else:
            # счиатем btime в unix time
            end_dt = (datetime.datetime.fromtimestamp(int(self.lbtime)) +
                  relativedelta(months=PRODUCT_DATA[self.pid][PERIOD][MONTH]))
            self.btime = str(int((end_dt - datetime.datetime(1970,1,1)).total_seconds()))
        self.oid = args[0]
        self.auto = 1 if args[1][1:-1] == 'subs' else 0
        self.uid = args[3]


class MPFSMissingServicesFinder(object):

    def __init__(self):
        pass

    def find_service(self, bb_record):
        bb_order = BBOrderInfo(bb_record)
        if int(bb_order.btime) < ctimestamp():
            # Эта услуга уже истекла, для нее ничего не ищем
            return

        # Ищем Order в MPFS базе
        order = billing_orders_history.find_one({'number': bb_order.oid})
        if not order:
            order = billing_orders_history.find_one({'_id': bb_order.oid})
        if not order:
            order = billing_orders.find_one({'_id': bb_order.oid})
        if not order:
            # Не нашли Order
            log.info('msg=%s\tuid=%s\toid=%s\tpid=%s\tctime=%s\tlbtime=%s' %
                     ('ORDER_NOT_FOUND', bb_order.uid, bb_order.oid, bb_order.pid, bb_order.ctime, bb_order.lbtime))
            return

        # Ищем Service в MPFS, который соответствует сервису из ББ
        if 'sid' in order:
            service = billing_services.find_one({'_id': order.get('sid')})
        else:
            log.info('msg=%s\tuid=%s\toid=%s' %
                     ('ORDER_HAS_NO_SID', bb_order.uid, bb_order.oid, ))
            service = billing_services.find_one({'order': bb_order.oid, 'uid': bb_order.uid})

        if not service:
            # Не нашли сервис в базе MPFS
            log.info('msg=%s\tuid=%s\toid=%s\tpid=%s\tctime=%s\tlbtime=%s\tbtime=%s' %
                     ('SERVICE_NOT_FOUND', bb_order.uid, bb_order.oid, bb_order.pid, bb_order.ctime, bb_order.lbtime,
                      bb_order.btime))

if __name__ == "__main__":
    finder = MPFSMissingServicesFinder()
    for bb_record in sys.stdin:
        finder.find_service(bb_record)
