#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Данный скрипт содержит множество подкоманд разного назначения
"""
import os
import inspect
import sys
import json
import random

from argparse import ArgumentParser

import mpfs.engine.process
if os.geteuid() != 0:
    mpfs.engine.process.setup_anyone_script()
else:
    mpfs.engine.process.setup_admin_script()

import requests

from mpfs.core.bus import Bus
from mpfs.core.services.passport_service import passport
from mpfs.core.address import Address
from mpfs.metastorage.mongo.binary import Binary
from mpfs.metastorage.mongo.collections.all_user_data import AllUserDataCollection
from mpfs.metastorage.mongo.util import decompress_data


class BaseSubScript(object):
    """
    Базовый класс для подпрограмм
    """
    @classmethod
    def get_argumentparser_args(cls):
        result = {
            'description': '',
            'help': '',
        }
        try:
            title, body = [i.strip() for i in cls.__doc__.strip().split("\n", 1)]
            result['description'] = body
            result['help'] = title
        except Exception:
            pass
        return result

    @staticmethod
    def fill_parser(parser):
        return parser

    @staticmethod
    def run(args_dict):
        raise NotImplementedError()


class Login2Uid(BaseSubScript):
    """
    Преобразует логины в uid-ы

    Получает список login-ов в STDIN, выводит результат в STDOUT.
    Если не может разрезолвить - падает
    """
    @staticmethod
    def run(args_dict):
        for line in sys.stdin:
            line = line.strip()
            user_info = passport.userinfo(login=line)
            if user_info.get('uid'):
                print user_info['uid']
            else:
                print >> sys.stderr, "Can't resolve login '%s'" % line
                exit(1)


class ProvideService(BaseSubScript):
    """
    Выдает услугу uid-у

    Получает список uid-ов в STDIN. Выводит uid-ы, которым предоставлена услуга.
    Если не может выдать место - падает
    """
    @staticmethod
    def fill_parser(parser):
        parser.add_argument('-p', help='product(pid)', required=True, dest='product')

    @staticmethod
    def run(args_dict):
        product = args_dict['product']

        for line in sys.stdin:
            uid = line.strip()
            if not uid.isdigit():
                print >> sys.stderr, "Not digit uid '%s'" % uid
                exit(1)
            else:
                params = {
                    'uid': uid,
                    'pid': product,
                    'line': 'bonus',
                    'ip': '127.0.0.1',
                }
                resp = requests.get("http://localhost/billing/service_create", params=params)
                if resp.status_code == requests.codes.ok or 'BillingProductIsSingletone' in resp.content:
                    print uid
                else:
                    print >> sys.stderr, uid


class DocsByHid(BaseSubScript):
    """
    Ищет все документы по hid-у

    Бежит по всем шардам и коллекциям с пользовательскими данными
    """
    @staticmethod
    def fill_parser(parser):
        parser.add_argument('hid', nargs=1, type=str, help="hid")

    @staticmethod
    def find_everywhere_and_print(spec):
        db_results = AllUserDataCollection().find(spec)
        for db_result in db_results:
            result = {
                '_shard_name': db_result.shard_name,
                '_collection_name': db_result.collection_name,
            }
            result.update(db_result.record)
            if 'zdata' in result:
                result['zdata'] = decompress_data(result['zdata'])
            if 'hid' in result:
                result['hid'] = str(result['hid'])
            print json.dumps(result)

    @staticmethod
    def run(args_dict):
        hid = args_dict['hid'][0]
        spec = {'hid': Binary(hid, subtype=2)}
        DocsByHid.find_everywhere_and_print(spec)


class DocsByStid(BaseSubScript):
    """
    Ищет все документы по stid-у

    Бежит по всем шардам и коллекциям с пользовательскими данными
    Поиск stid-а осуществляется в 'data.stids'
    """
    @staticmethod
    def fill_parser(parser):
        parser.add_argument('stid', nargs=1, type=str, help="stid")

    @staticmethod
    def run(args_dict):
        stid = args_dict['stid'][0]
        spec = {'data.stids': {
            '$elemMatch': {'stid': stid}}
        }
        DocsByHid.find_everywhere_and_print(spec)


class EncodeStid(BaseSubScript):
    """
    Cкрипт-шифратор стидов

    Cкрипт-шифратор стидов
    """
    @staticmethod
    def fill_parser(parser):
        parser.add_argument('-s', help='string stid', required=True, dest='stid')

    @staticmethod
    def run(args_dict):
        mid = args_dict['stid']
        from mpfs.core.services.zaberun_service import Zaberun
        z = Zaberun()
        processed_mid = z.crypt_agent.encrypt(mid)
        import re
        processed_mid = re.sub(r'\+', r'-', processed_mid)
        processed_mid = re.sub(r'\/', r'_', processed_mid)
        import urllib
        print urllib.quote(processed_mid, safe="")


if __name__ == '__main__':
    parser = ArgumentParser(description=__doc__)
    sub_parser = parser.add_subparsers(dest='cmd_name')

    sub_scripts = {}

    for name, cls in globals().copy().iteritems():
        if (inspect.isclass(cls) and
                issubclass(cls, BaseSubScript) and
                cls is not BaseSubScript):
            sub_cmd_parser = sub_parser.add_parser(name, **cls.get_argumentparser_args())
            cls.fill_parser(sub_cmd_parser)
            sub_scripts[name] = cls

    namespace = vars(parser.parse_args())
    sub_scripts[namespace['cmd_name']].run(namespace)
