# -*- coding: utf-8 -*-

from functools import update_wrapper

from django.conf import settings
from django.contrib.admin import AdminSite
from django.contrib.auth.models import Group
from django.http import HttpResponseForbidden, Http404
from django.shortcuts import redirect
from django.views.decorators.cache import never_cache
from django.views.decorators.csrf import csrf_protect

from django_yauth.util import get_passport_url


class BaseAdminSite(AdminSite):
    """Кастомный класс сайта админки.

    Пускает только пользователей с проставленным флагом `is_staff`,
    иначе возвращает 403 ошибку.
    На URL логина возвращает 404 ошибку, тк логин происходит через Я.Паспорт.
    Для того, чтобы пользователь видел таблицы в админке он должен быть либо суперюзером,
    либо иметь права на эти таблицы, выданные через стандартный механизм пермишеннов. По сути
    IDM это и делает через http API.
    """

    def admin_view(self, view, cacheable=False):
        def inner(request, *args, **kwargs):
            if not self.has_permission(request):
                return HttpResponseForbidden()
            return view(request, *args, **kwargs)

        if not cacheable:
            inner = never_cache(inner)
        if not getattr(view, 'csrf_exempt', False):
            inner = csrf_protect(inner)
        return update_wrapper(inner, view)

    def login(self, request, extra_context=None):
        raise Http404()

    def logout(self, request, extra_context=None):
        return redirect(get_passport_url('delete'))

    def has_permission(self, request):
        return request.user.is_superuser

site = BaseAdminSite()
site.site_header = settings.ADMIN_SITE_HEADER

site.register(Group)
