# -*- coding: utf-8 -*-

from __future__ import unicode_literals
from collections import Counter

from django import forms
from django.core.exceptions import ValidationError

from api_admin.api_auth.validators import validate_ip_range
from mpfs.common.util import from_json


class BaseListField(object):
    DEFAULT_SEPARATOR = '\r\n'

    def __init__(self, *args, **kwargs):
        super(BaseListField, self).__init__()
        if 'separator' in kwargs:
            assert isinstance(kwargs['separator'], (str, unicode))
            self.separator = kwargs['separator']
        else:
            self.separator = self.DEFAULT_SEPARATOR


class StringListField(forms.CharField, BaseListField):

    def prepare_value(self, value):
        """
        Принимает на вход список строк и преобразует его в строку.

        Это значение в конечном счете уйдет в виджет.
        """
        if value is None:
            return value

        if isinstance(value, (str, unicode)):
            # к примеру если форма невалидна и мы ее обратно шлем
            return value
        elif isinstance(value, (list, tuple)):
            return self.separator.join(value)
        else:
            raise NotImplementedError()

    def to_python(self, value):
        """Разбивает строку по символу перевода строки и формирует список из полученных строк."""
        value = super(StringListField, self).to_python(value)
        if not value:
            return []
        return value.split(self.separator)


class IntegerListField(forms.IntegerField, BaseListField):

    def prepare_value(self, value):
        """
        Принимает на вход список интов и преобразует его в строку.

        Это значение в конечном счете уйдет в виджет.
        """
        if value is None:
            return value

        if isinstance(value, (str, unicode)):
            # к примеру если форма невалидна и мы ее обратно шлем
            return value
        elif isinstance(value, int):
            return str(value)
        elif isinstance(value, (list, tuple)):
            str_values = [str(i) for i in value]
            return self.separator.join(str_values)
        else:
            raise NotImplementedError()

    def to_python(self, value):
        """Разбивает строку по символу перевода строки и формирует список из полученных интов."""
        values = value.split(self.separator)
        int_values = [super(IntegerListField, self).to_python(v) for v in values if v not in self.empty_values]
        if not int_values:
            return []
        return int_values

    def validate(self, value):
        if not isinstance(value, list):
            raise ValidationError('IntegerListField must be list')
        for i in value:
            if not isinstance(i, int):
                raise ValidationError('IntegerListField\'s values must be ints')


class IPRangeListField(StringListField):
    def validate(self, value):
        """Проверяет, что диапазоны адресов заданы в правильном формате.

        Форматы:
          * единичный адрес: `192.168.0.1`;
          * адрес подсети: `192.168.0.0/24`;
          * диапазон адресов: `192.168.0.5 - 192.168.0.19`.

        """
        super(IPRangeListField, self).validate(value)

        for ip_range in value:
            validate_ip_range(ip_range)


class JsonTextField(forms.CharField):
    def validate(self, value):
        super(JsonTextField, self).validate(value)
        try:
            return from_json(value)
        except Exception:
            raise ValidationError('Invalid JSON')


class ConfigTextField(JsonTextField):
    def validate(self, value):
        from api_admin.api_auth.forms import AuthClientAddForm

        validated_value = super(ConfigTextField, self).validate(value)
        if not isinstance(validated_value, list):
            raise ValidationError('ConfigTextField accepts JSONs presented as list only')

        for client_config in validated_value:
            if not isinstance(client_config, dict):
                raise ValidationError('ConfigTextField\'s clients should be dictionaries')
            AuthClientAddForm.validate_all_for_client_config(client_config)

        client_occurences = Counter([x['name'] for x in validated_value])
        duplicate_clients = [k for k, v in client_occurences.items() if v > 1]
        if duplicate_clients:
            raise ValidationError('ConfigTextField has duplicates "%s"' % duplicate_clients)
