# -*- coding: utf-8 -*-

from __future__ import unicode_literals
from itertools import chain

from django import forms
from django.conf import settings
from django.core.exceptions import ValidationError
from django.core.validators import RegexValidator
from django.utils.functional import cached_property

from api_admin.api_auth.validators import ListValueValidator
from api_admin.zookeeper import get_or_create_client
from api_admin.api_auth.fields import ConfigTextField, IntegerListField, IPRangeListField, StringListField
from api_admin.api_auth.models import Scope

from mpfs.common.util import from_json


class BaseForm(forms.Form):
    pass


class AuthClientAddForm(BaseForm):
    """Форма добавления клиента API.

    При изменении имён полей обязательно необходимо удостовериться, что медиа-файлы
    продолжают корректно работать.
    """
    METHOD_TOKEN = 'token'
    METHOD_COOKIE = 'cookie'
    METHOD_TVM = 'tvm'
    METHOD_YATEAM_TVM = 'yateam_tvm'
    METHOD_TVM_2_0 = 'tvm_2_0'
    METHOD_TVM_2_0_TICKET_ONLY = 'tvm_2_0_tickets_only'
    METHOD_EXT_TOKEN = 'ext_tokens'
    METHOD_OAUTH = 'oauth'
    METHOD_YATEAM_OAUTH = 'yateam_oauth'
    METHODS_CHOICES = (
        (METHOD_TOKEN, 'Токен (внутреннее api)'),
        (METHOD_COOKIE, 'Куки'),
        (METHOD_TVM, 'TVM'),
        (METHOD_YATEAM_TVM, 'TVM (yateam)'),
        (METHOD_TVM_2_0, 'TVM 2.0'),
        (METHOD_TVM_2_0_TICKET_ONLY, 'TVM 2.0 только UserTicket тикет'),
        (METHOD_EXT_TOKEN, 'Токен (внешнее api)'),
        (METHOD_OAUTH, 'OAuth'),
        (METHOD_YATEAM_OAUTH, 'OAuth (yateam)'),
    )
    enabled = forms.BooleanField(label='Активен', initial=True, required=False)
    name = forms.CharField(
        label='Имя', max_length=64,
        help_text=(
            'Это просто уникальный идентификатор приложения в API. '
            'Используется только для того, чтобы отличать одного клиента от другого и более ни для чего.'
        )
    )
    auth_methods = forms.MultipleChoiceField(
        label='Разрешенные методы',
        choices=METHODS_CHOICES,
        widget=forms.CheckboxSelectMultiple()
    )
    oauth_client_name = forms.CharField(label='Имя OAuth-клиента', max_length=64)
    oauth_client_id = forms.CharField(
        label='Идентификатор клиента', max_length=64,
        help_text=(
            'Для OAuth-клиентов - это OAuth Client ID, '
            'а для других клиентов - просто произвольный уникальный идентификатор.'
        )
    )
    limit_groups = StringListField(
        label='Лимиты клиента (группы в рейтлимитере)', max_length=512, widget=forms.Textarea(), required=False,
        help_text='В формате group_name=multiplier, где multiplier - сколько тысяч rps разрешено для группы group_name',
        validators=[
            ListValueValidator(RegexValidator(
                regex=r'^[a-zA-Z_-]+=\d+$',
                message='В формате group_name=multiplier, group_name = [a-zA-Z_]+, multiplier = \\d+'
            ))
        ]
    )
    oauth_scopes = forms.ModelMultipleChoiceField(
        label='Права',
        required=False,
        queryset=Scope.objects.all(),
        to_field_name='key',
        widget=forms.CheckboxSelectMultiple(),
        help_text=('Права, которыми будет обладать клиент, '
                   'авторизуясь через Cookie или через внутреннюю авторизацию в API.')
    )
    token = forms.CharField(label='Токен (внутреннее api)', max_length=64, required=False)
    tvm_client_ids = IntegerListField(
        label='Идентификаторы клиентов TVM', widget=forms.Textarea(),
        help_text='По строке на client_id.', required=False
    )
    tvm_2_0_client_ids = IntegerListField(
        label='Идентификаторы клиентов TVM 2.0', widget=forms.Textarea(),
        help_text='По строке на client_id.', required=False
    )
    cookie_auth_client_id = forms.CharField(
        label='Client ID, который будет соответствовать клиенту при авторизации по Cookie',
        max_length=64, required=False
    )
    allowed_origin_hosts = StringListField(
        label='Список разрешенных хостов', max_length=2048, widget=forms.Textarea(),
        help_text='По строке на одно правило. Поддерживаются RegExp.', required=False
    )
    ext_tokens = StringListField(
        label='Список токенов (для внешнего api)', max_length=512, widget=forms.Textarea(),
        help_text='По строке на один токен.', required=False
    )
    allowed_network_addresses = IPRangeListField(
        label='Список адресов, с которых клиенту разрешено делать запросы.',
        max_length=4096,
        widget=forms.Textarea(),
        help_text='По строке на адрес. Примеры: единичный адрес: 192.168.0.1; адрес подсети: 192.168.0.0/24; диапазон '
                  'адресов: 192.168.0.5 - 192.168.0.19.',
        required=False
    )

    AUTH_METHODS_REQUIRED_FIELDS = {
        METHOD_TOKEN: ('token',),
        METHOD_COOKIE: ('cookie_auth_client_id',),
        METHOD_TVM: ('tvm_client_ids',),
        METHOD_YATEAM_TVM: ('tvm_client_ids',),
        METHOD_TVM_2_0: ('tvm_2_0_client_ids',),
        METHOD_TVM_2_0_TICKET_ONLY: ('tvm_2_0_client_ids',),
        METHOD_EXT_TOKEN: ('ext_tokens',)
    }

    class Media:
        js = ('api_auth/js/client-add.js',)

    def clean_name(self):
        """Провалидировать имя.

        Проверяет что еще не существует клиент с переданным именем.
        """
        name = self.cleaned_data['name']

        # проверяем существует ли клиент с переданным именем
        if name in {client['name'] for client in self._zk_auth_settings}:
            raise forms.ValidationError('Клиент с таким именем уже существует.')

        return name

    def clean_oauth_scopes(self):
        oauth_scopes = self.cleaned_data.get('oauth_scopes')
        if oauth_scopes is not None:
            return map(lambda x: x.key, oauth_scopes)

        return oauth_scopes

    def clean_tvm_2_0_client_ids(self):
        current_name = self.cleaned_data['name']
        cleaned_ids = set(self.cleaned_data['tvm_2_0_client_ids'])
        overlapping_ids = cleaned_ids.intersection(chain.from_iterable(
            client.get('tvm_2_0_client_ids', ())
            for client in self._zk_auth_settings
            if client['name'] != current_name
        ))
        if overlapping_ids:
            raise forms.ValidationError('%s уже заняты: %s' % (
                self.base_fields['tvm_2_0_client_ids'].label,
                ', '.join(map(str, sorted(overlapping_ids)))
            ))

        return self.cleaned_data['tvm_2_0_client_ids']

    def clean_limit_groups(self):
        result = []
        limit_groups = self.cleaned_data.get('limit_groups')

        if limit_groups:
            for limit_group in limit_groups:
                limit = list(map(lambda s: s.strip(), limit_group.split('=')))
                if len(limit) == 2:
                    result.append({'name': limit[0], 'multiplier': int(limit[1])})

        return result

    def clean(self):
        form_data = self.cleaned_data
        form_data, errors = self.validate_client_auth_requirements(form_data)
        for required_field_name, error_msg in errors:
            self.add_error(
                required_field_name,
                error_msg
            )
        return form_data

    @cached_property
    def _zk_auth_settings(self):
        zk_client = get_or_create_client()
        value, _ = zk_client.get(settings.ZOOKEEPER_PLATFORM_AUTH_SETTINGS_PATH)
        return from_json(value)

    @classmethod
    def validate_client_auth_requirements(cls, client_config):
        """
        Свалидировать конфиг клиента

        :param dict client_config:
        :return: (<конфиг после обработки>, <список ошибок>)
        """
        errors = []
        auth_methods = client_config.get('auth_methods', tuple())
        auth_labels = dict(cls.METHODS_CHOICES)

        if cls.METHOD_TVM in auth_methods and cls.METHOD_YATEAM_TVM in auth_methods:
            errors.append(('auth_methods',
                           'Только один из %s, %s метод может быть определен для клиента.' % (
                               auth_labels[cls.METHOD_YATEAM_TVM],
                               auth_labels[cls.METHOD_TVM]
                           )))

        for auth_method in auth_methods:
            required_fields = cls.AUTH_METHODS_REQUIRED_FIELDS.get(auth_method, tuple())
            for required_field_name in required_fields:
                if not client_config.get(required_field_name):
                    errors.append((required_field_name,
                                   'Для авторизации через %s поле «%s» является обязательным для заполнения.' % (
                                       auth_labels[auth_method],
                                       cls.declared_fields[required_field_name].label
                                   )))

        if 'tvm_client_ids' in client_config:
            if not client_config['tvm_client_ids']:
                del client_config['tvm_client_ids']

        if 'tvm_2_0_client_ids' in client_config:
            if not client_config['tvm_2_0_client_ids']:
                del client_config['tvm_2_0_client_ids']

        if 'cookie_auth_client_id' in client_config:
            if not client_config['cookie_auth_client_id']:
                del client_config['cookie_auth_client_id']

        return client_config, errors

    @classmethod
    def validate_client_config_structure(cls, client_config):
        required_fields = [k for k, v in cls.declared_fields.items() if v.required]
        for required_field in required_fields:
            if required_field not in client_config:
                raise ValidationError('Fields "%s" is missing' % required_field)

        for field_name, _ in client_config.items():
            if field_name not in cls.declared_fields:
                raise ValidationError('Unexpected field name "%s" of client "%s"' % (field_name, client_config['name']))

    @classmethod
    def validate_config_fields(cls, client_config):
        for field, value in client_config.items():
            cls.declared_fields[field].validate(value)

    @classmethod
    def validate_all_for_client_config(cls, client_config):
        cls.validate_client_config_structure(client_config)
        client_config, errors = cls.validate_client_auth_requirements(client_config)
        if errors:
            # Сообщаем только о первой ошибке
            required_field_name, auth_method_name = errors[0]
            raise ValidationError(
                'Field "%s" is required for "%s" authorization' %
                (required_field_name, auth_method_name)
            )
        cls.validate_config_fields(client_config)


class AuthClientEditForm(AuthClientAddForm):
    """Форма редактирования клиента API."""

    def __init__(self, *args, **kwargs):
        super(AuthClientAddForm, self).__init__(*args, **kwargs)
        self.fields['name'].widget.attrs = {'readonly': 'readonly'}

    def clean_name(self):
        return self.cleaned_data['name']


class AuthRulesUploadForm(BaseForm):
    """Форма загрузки конфигурации через JSON."""
    json_config = ConfigTextField(
        label='Конфиг для загрузки', widget=forms.Textarea(),
        help_text='Список в формате JSON', required=True
    )
