# -*- coding: utf-8 -*-
from django.core.exceptions import ValidationError
from django.core.validators import validate_ipv4_address
from django.utils.ipv6 import is_valid_ipv6_address
from django.utils.translation import ugettext_lazy

IP_BLOCK_SEPARATOR = '-'
NETWORK_MASK_SEPARATOR = '/'
IPV4_ADDRESS_BITS_COUNT = 32
IPV6_ADDRESS_BITS_COUNT = 128


def is_valid_ipv4_address(value):
    """Проверяет валидность IPv4 адреса."""
    try:
        validate_ipv4_address(value)
    except ValidationError:
        return False
    else:
        return True


def is_valid_ipv46_address(value):
    """Проверяет валидность IPv4 и IPv6 адресов."""
    return (is_valid_ipv4_address(value) or
            is_valid_ipv6_address(value))


def validate_ip_block(value):
    """Проверяет валидность IPv4 и IPv6 блока адресов."""
    if value.count(IP_BLOCK_SEPARATOR) != 1:
        raise ValidationError(ugettext_lazy('Multiple separators in IP addresses block (should be only one): '
                                            '%(address)s.'),
                              params={'address': value},
                              code='invalid')

    start, end = value.split(IP_BLOCK_SEPARATOR)
    start = start.strip()
    end = end.strip()
    is_valid_ipv4_range = (is_valid_ipv4_address(start) and
                           is_valid_ipv4_address(end))
    is_valid_ipv6_range = (is_valid_ipv6_address(start) and
                           is_valid_ipv6_address(end))

    if not (is_valid_ipv4_range or
            is_valid_ipv6_range):
        raise ValidationError(ugettext_lazy('Addresses should be both IPv4 or both IPv6: %(address)s.'),
                              params={'address': value},
                              code='invalid')


def validate_network_address(value):
    # диапазон адресов
    if value.count(NETWORK_MASK_SEPARATOR) != 1:
        raise ValidationError(ugettext_lazy('Multiple network mask separators in IP network address '
                                            '(should be only one): %(address)s.'),
                              params={'address': value},
                              code='invalid')

    address, mask = value.split(NETWORK_MASK_SEPARATOR)
    mask = int(mask)
    is_valid_ipv4_network = (is_valid_ipv4_address(address) and
                             0 <= mask <= IPV4_ADDRESS_BITS_COUNT)
    is_valid_ipv6_network = (is_valid_ipv6_address(address) and
                             0 <= mask <= IPV6_ADDRESS_BITS_COUNT)
    if not (is_valid_ipv4_network or
            is_valid_ipv6_network):
        raise ValidationError(ugettext_lazy('Invalid IP network address: %(address)s.'),
                              params={'address': value},
                              code='invalid')


def validate_ip_range(value):
    """Проверяет валидность IPv4 и IPv6 диапазоны адресов."""
    if IP_BLOCK_SEPARATOR in value:
        validate_ip_block(value)
    elif NETWORK_MASK_SEPARATOR in value:
        validate_network_address(value)
    else:
        # Значение не является ни блоком адресов, ни адресом сети.
        # Проверяем последний возможный формат: единичный адрес
        if not is_valid_ipv46_address(value):
            raise ValidationError(ugettext_lazy('Invalid format for IP addresses range: %(address)s.'),
                                  params={'address': value},
                                  code='invalid')


class ListValueValidator(object):
    def __init__(self, validator):
        self.validator = validator

    def __call__(self, value):
        for val in value:
            self.validator(val)
