# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import datetime
import json

from django.conf import settings
from django.contrib import messages
from django.core.paginator import Paginator, EmptyPage, PageNotAnInteger
from django.utils.decorators import method_decorator
from django.http import HttpResponse, Http404
from django.shortcuts import redirect, render
from django.core.urlresolvers import reverse
from django.views.generic import View

from mpfs.core.zookeeper.shortcuts import get_checksum
from api_admin.zookeeper import client as zk_client
from api_admin.api_auth.forms import AuthClientAddForm, AuthClientEditForm, AuthRulesUploadForm
from api_admin.api_auth.models import AuthRulesBackup
from api_admin.signals import pre_config_changes
from api_admin.site_auth import roles
from api_admin.site_auth.decorators import auth_check

from mpfs.common.util import from_json, to_json


def _load_auth_clients_data():
    """Получить данные всех клиентов API."""
    value, stats = zk_client.get(settings.ZOOKEEPER_PLATFORM_AUTH_SETTINGS_PATH)
    sorted_data = sorted(from_json(value), key=lambda x: x['name'])
    return value, sorted_data


class AuthClientListView(View):
    """Вывод списка всех клиентов API."""
    @method_decorator(
        auth_check(
            role_names=[
                roles.ROLE_NAME_VIEWER,
                roles.ROLE_NAME_EDITOR,
                roles.ROLE_NAME_SUPERUSER
            ]
        )
    )
    def get(self, request, *args, **kwargs):
        raw_data, data = _load_auth_clients_data()
        checksum = get_checksum(data)
        obj_list = []
        view = request.GET.get('view', 'table')
        for obj in data:
            obj['edit_url'] = reverse('api_auth:client-detail-edit', args=[obj['name']])
            obj['detail_view_url'] = reverse('api_auth:client-detail', args=[obj['name']])
            obj['delete_url'] = reverse('api_auth:client-detail', args=[obj['name']])
            obj_list.append(obj)
        if view == 'table':
            template = 'api_auth/auth-client-list-table-view.html'
        elif view == 'json':
            template = 'api_auth/auth-client-list-json-view.html'
        else:
            raise NotImplementedError()
        return render(
            request,
            template, {
                'obj_list': obj_list,
                'field_list': [
                    ('name', AuthClientAddForm.declared_fields['name'].label),
                    ('auth_methods', AuthClientAddForm.declared_fields['auth_methods'].label),
                    ('oauth_scopes', AuthClientAddForm.declared_fields['oauth_scopes'].label),
                ],
                'title': 'Клиенты (v: %s)' % checksum,
            }
        )

    @method_decorator(
        auth_check(
            role_names=[
                roles.ROLE_NAME_EDITOR,
                roles.ROLE_NAME_SUPERUSER
            ]
        )
    )
    def post(self, request, *args, **kwargs):
        action = request.POST.get('action')
        if action != 'create_backup':
            raise NotImplementedError
        _, data = _load_auth_clients_data()
        pre_config_changes.send(sender=None, user=request.user, config_dict=data)
        return redirect('api_auth:client-list')


class AuthClientDetailView(View):
    """Ресурс конкретного клиента API."""
    @method_decorator(
        auth_check(
            role_names=[
                roles.ROLE_NAME_VIEWER,
                roles.ROLE_NAME_EDITOR,
                roles.ROLE_NAME_SUPERUSER
            ]
        )
    )
    def get(self, request, *args, **kwargs):
        name = self.kwargs['id']
        raw_data, data = _load_auth_clients_data()
        for client in data:
            if client['name'] == name:
                break
        else:
            raise Http404()

        return render(request, 'api_auth/auth-client-detail.html', {
            'title': 'Клиент «%s»' % client['name'],
            'obj': client,
            'field_list': [(key, field.label) for key, field in AuthClientAddForm.declared_fields.items()],
            'edit_url': reverse('api_auth:client-detail-edit', args=[name])
        })

    @method_decorator(
        auth_check(
            role_names=[
                roles.ROLE_NAME_EDITOR,
                roles.ROLE_NAME_SUPERUSER
            ]
        )
    )
    def delete(self, request, *args, **kwargs):
        name = self.kwargs['id']
        raw_data, data = _load_auth_clients_data()
        if name not in {client['name'] for client in data}:
            raise Http404()

        zk_client.set(
            settings.ZOOKEEPER_PLATFORM_AUTH_SETTINGS_PATH,
            to_json([client for client in data if client['name'] != name]).encode('utf8'),
            request.user
        )
        if request.is_ajax():
            messages.add_message(request, messages.SUCCESS, 'Клиент «%s» удалён.' % name)
            return HttpResponse()
        else:
            return redirect('api_auth:client-list')


class AuthClientEditView(View):
    """Редактирование информации о конкретном клиенте API."""
    @method_decorator(
        auth_check(
            role_names=[
                roles.ROLE_NAME_EDITOR,
                roles.ROLE_NAME_SUPERUSER
            ]
        )
    )
    def get(self, request, *args, **kwargs):
        raw_data, data = _load_auth_clients_data()
        name = self.kwargs['id']
        for client_data in data:
            if client_data.get('name') != name:
                continue
            else:
                self.prepare_client_data(client_data)
                form = AuthClientEditForm(initial=client_data)
                return HttpResponse(
                    content=render(
                        request, 'api_auth/auth-client-edit.html',
                        {'form': form}
                    )
                )
        raise Http404()

    @staticmethod
    def prepare_client_data(client_data):
        limit_groups = client_data.get('limit_groups')
        if limit_groups:
            string_limit_groups = []
            for limit_group in limit_groups:
                string_limit_groups.append(u'%(name)s=%(multiplier)s' % limit_group)

            client_data['limit_groups'] = '\n'.join(string_limit_groups)

    @method_decorator(
        auth_check(
            role_names=[
                roles.ROLE_NAME_EDITOR,
                roles.ROLE_NAME_SUPERUSER
            ]
        )
    )
    def post(self, request, *args, **kwargs):
        action = request.POST.get('action')
        name = self.kwargs['id']
        if action == 'delete':
            raw_data, data = _load_auth_clients_data()
            if name not in {client['name'] for client in data}:
                raise Http404()

            zk_client.set(
                settings.ZOOKEEPER_PLATFORM_AUTH_SETTINGS_PATH,
                to_json([client for client in data if client['name'] != name]).encode('utf8'),
                request.user
            )
            return redirect('api_auth:client-list')
        elif action == 'save':
            form = AuthClientEditForm(request.POST)
            if not form.is_valid():
                return HttpResponse(
                    content=render(
                        request, 'api_auth/auth-client-edit.html',
                        {'form': form}
                    )
                )
            else:
                client_data = form.cleaned_data
                # изменяем данные этого клиента
                value, stats = zk_client.get(settings.ZOOKEEPER_PLATFORM_AUTH_SETTINGS_PATH)
                data = from_json(value)
                data = [client for client in data if client['name'] != client_data['name']]
                data.append(client_data)
                zk_client.set(
                    settings.ZOOKEEPER_PLATFORM_AUTH_SETTINGS_PATH,
                    to_json(data).encode('utf8'),
                    request.user
                )

                return redirect('api_auth:client-detail', name)
        elif action == 'cancel':
            return redirect('api_auth:client-detail-edit', name)
        else:
            raise NotImplementedError


class AuthClientAddView(View):
    """Добавление нового клиента API."""
    @method_decorator(
        auth_check(
            role_names=[
                roles.ROLE_NAME_EDITOR,
                roles.ROLE_NAME_SUPERUSER
            ]
        )
    )
    def get(self, request, *args, **kwargs):
        form = AuthClientAddForm()
        return HttpResponse(
            content=render(
                request, 'api_auth/auth-client-add.html',
                {'form': form}
            )
        )

    @method_decorator(
        auth_check(
            role_names=[
                roles.ROLE_NAME_EDITOR,
                roles.ROLE_NAME_SUPERUSER
            ]
        )
    )
    def post(self, request, *args, **kwargs):
        form = AuthClientAddForm(request.POST)
        # проверка на существование клиента произойдет при валидации формы
        if not form.is_valid():
            return HttpResponse(
                content=render(
                    request, 'api_auth/auth-client-add.html',
                    {'form': form}
                )
            )
        else:
            client_data = form.cleaned_data

            # добавляем данные нового клиента
            # в случае если уже существует клиент с тем же именем,
            # то вызовет ошибку
            value, stats = zk_client.get(settings.ZOOKEEPER_PLATFORM_AUTH_SETTINGS_PATH)
            data = from_json(value)
            if client_data['name'] in {client['name'] for client in data}:
                return HttpResponse(status=400)
            data.append(client_data)
            zk_client.set(settings.ZOOKEEPER_PLATFORM_AUTH_SETTINGS_PATH, to_json(data).encode('utf8'), request.user)

            return redirect('api_auth:client-detail-edit', client_data['name'])


class AuthRulesBackupListView(View):
    """Листинг бекапов"""

    @method_decorator(
        auth_check(
            role_names=[
                roles.ROLE_NAME_VIEWER,
                roles.ROLE_NAME_EDITOR,
                roles.ROLE_NAME_SUPERUSER
            ]
        )
    )
    def get(self, request, *args, **kwargs):
        ordered_backups = AuthRulesBackup.objects.order_by('-ctime')
        paginator = Paginator(ordered_backups, 15)
        page = request.GET.get('page')
        try:
            paginated_backups = paginator.page(page)
        except PageNotAnInteger:
            paginated_backups = paginator.page(1)
        except EmptyPage:
            paginated_backups = paginator.page(paginator.num_pages)

        obj_list = []
        template = 'api_auth/auth-backup-list.html'
        for obj in paginated_backups:
            obj_list.append({
                'user': obj.user.get_username(),
                'ctime': obj.ctime,
                'detail_view_url': reverse('api_auth:auth-backup-details', args=[obj.checksum]),
                'checksum': obj.checksum,
            })
        return render(
            request,
            template, {
                'obj_list': obj_list,
                'field_list': [
                    ('user', 'Имя'),
                    ('ctime', 'Время создания бэкапа'),
                ],
                'title': 'Список бекапов',
                'paginated_list': paginated_backups,
            }
        )


class AuthRulesBackupDetailView(View):
    """Работа с бэкапом"""
    @method_decorator(
        auth_check(
            role_names=[
                roles.ROLE_NAME_VIEWER,
                roles.ROLE_NAME_EDITOR,
                roles.ROLE_NAME_SUPERUSER
            ]
        )
    )
    def get(self, request, *args, **kwargs):
        backup = AuthRulesBackup.objects.filter(checksum=kwargs['checksum']).first()
        if backup is None:
            return render(request, 'common/404.html')

        template = 'api_auth/auth-backup-details.html'
        obj_list = sorted(backup.data, key=lambda x: x['name'])
        pretty_json = json.dumps(obj_list, indent=4)
        return render(
            request,
            template, {
                'obj_list': obj_list,
                'field_list': [
                    ('name', AuthClientAddForm.declared_fields['name'].label),
                    ('auth_methods', AuthClientAddForm.declared_fields['auth_methods'].label),
                    ('oauth_scopes', AuthClientAddForm.declared_fields['oauth_scopes'].label),
                ],
                'checksum': backup.checksum,
                'pretty_json': pretty_json,
            }
        )

    @method_decorator(
        auth_check(
            role_names=[
                roles.ROLE_NAME_EDITOR,
                roles.ROLE_NAME_SUPERUSER
            ]
        )
    )
    def post(self, request, *args, **kwargs):
        action = request.POST.get('action')
        if action != 'restore':
            raise NotImplementedError

        backup = AuthRulesBackup.objects.filter(checksum=self.kwargs['checksum']).first()
        zk_client.set(
            settings.ZOOKEEPER_PLATFORM_AUTH_SETTINGS_PATH,
            to_json(backup.data).encode('utf8'),
            request.user
        )
        return redirect('api_auth:client-list')


class AuthRulesUploadView(View):
    """Загрузка конфигурации"""

    @method_decorator(
        auth_check(
            role_names=[
                roles.ROLE_NAME_EDITOR,
                roles.ROLE_NAME_SUPERUSER
            ]
        )
    )
    def get(self, request, *args, **kwargs):
        form = AuthRulesUploadForm()
        return HttpResponse(
            content=render(
                request, 'api_auth/auth-config-upload.html',
                {'form': form}
            )
        )

    @method_decorator(
        auth_check(
            role_names=[
                roles.ROLE_NAME_EDITOR,
                roles.ROLE_NAME_SUPERUSER
            ]
        )
    )
    def post(self, request, *args, **kwargs):
        action = request.POST.get('action')
        if action != 'upload':
            raise NotImplementedError
        form = AuthRulesUploadForm(request.POST)
        if not form.is_valid():
            return HttpResponse(
                content=render(
                    request, 'api_auth/auth-config-upload.html',
                    {'form': form}
                )
            )
        config = form.cleaned_data
        zk_client.set(
            settings.ZOOKEEPER_PLATFORM_AUTH_SETTINGS_PATH,
            config['json_config'].encode('utf8'),
            request.user
        )
        return redirect('api_auth:client-list')
