# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging
import logging.config
import signal

from mpfs.common.util.logger import rotate_logs
from api_admin.env import get_app_name, get_host_name


class CloudApiAdminLogger(logging.getLoggerClass()):
    _signal_safe = True
    _process_aware = True

    def makeRecord(self, name, level, fn, lno, msg, args, exc_info, func=None, extra=None):

        new_extra = {}
        mpfs_extra = {
            'caa_app_name': get_app_name() or '',
            'caa_host_name': get_host_name() or '',
        }

        if extra is not None:
            keys = mpfs_extra.viewkeys() & extra.viewkeys()
            if keys:
                raise KeyError("Attempt to overwrite %r in `caa_extra`" % keys)
            new_extra.update(extra)
        new_extra.update(mpfs_extra)
        return super(CloudApiAdminLogger, self).makeRecord(name, level, fn, lno, msg, args, exc_info, func, extra=new_extra)


def configure_logging(dict_config, signum=signal.SIGRTMIN):
    """Настроить подсистему логирования Админки API.

    Вызывается единственный раз в точке входа в приложение, до большинства импортов.
    Список выполняемых действий:
        * устанавливает класс логгера
        * настраивает `handlers`, `formatters`, `filters`, `loggers` через
            стандартную библиотеку :mod:`logging`
        * устанавливает ротацию по сигналу `signum`

    Полезные ссылки:
        `logging.dictConfig`
        https://docs.python.org/2/library/logging.config.html#dictionary-schema-details

    :type dict_config: dict
    :type app_name: None | str
    :type signum: int
    """

    if not isinstance(dict_config, dict):
        raise TypeError("`dict` is required, got `%s` instead" % type(dict_config))

    if not isinstance(signum, int):
        raise TypeError("`int` is required, got `%s` instead" % type(signum))

    logging.setLoggerClass(CloudApiAdminLogger)
    logging.config.dictConfig(dict_config)

    signal.signal(signum, rotate_logs)
    signal.siginterrupt(signum, False)


def get_access_log():
    return logging.getLogger('mpfs-api-admin.access')


def get_default_log():
    return logging.getLogger('mpfs-api-admin.default')


def get_error_log():
    return logging.getLogger('mpfs-api-admin.error')
