# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import hashlib
import re
import time
import traceback

from api_admin.env import generate_request_id, get_host_name, set_request_id, set_yandex_cloud_request_id
from api_admin.log import get_access_log, get_error_log


HTTP_HEADERS_REGEX = re.compile(r'^(HTTP_.+|CONTENT_TYPE|CONTENT_LENGTH)$')

access_log = get_access_log()
error_log = get_error_log()


class LogAccessMiddleware(object):
    def process_request(self, request):
        request.received_at = time.time()
        request.ycrid = '%s-%s' % (hashlib.md5().hexdigest(), get_host_name())
        set_yandex_cloud_request_id(request.ycrid)

    def process_response(self, request, response):
        request_headers = {}
        for header in request.META:
            if HTTP_HEADERS_REGEX.match(header) and header not in ('HTTP_COOKIE',):
                value = request.META[header]
                header = header.lstrip('HTTP_')
                header = header.replace('_', '-')
                header = '-'.join(
                    map(
                        lambda x: x.lower().capitalize(),
                        header.split('-')
                    )
                )
                request_headers[header] = value

        request_id = generate_request_id()
        set_request_id(request_id)

        ya_user = getattr(request, 'yauser', None)
        uid = ''
        if ya_user:
            uid = getattr(ya_user, 'uid', uid)

        extra = {
            'scheme': request.scheme,
            'status': response.status_code,
            'request_time': time.time() - request.received_at,
            'uri': request.get_full_path(),
            'caa_yandex_cloud_request_id': request.ycrid,
            'headers': ' '.join(['"%s: %s"' % (k, v) for k, v in request_headers.iteritems()]),
            'method': request.method,
            'uid': uid,
            'client_id': None,
            'proto': request.META['SERVER_PROTOCOL'],
            'request_id': request_id
        }
        access_log.info('', extra=extra)
        return response

    def process_exception(self, request, exception):
        error_log.error(traceback.format_exc())
        return None
