# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import os
import socket

import yenv

from django.conf import settings
from django.shortcuts import render
from django.http import HttpResponse, JsonResponse

from api_admin.site_auth import roles
from api_admin.site_auth.decorators import auth_check
from api_admin.zookeeper import get_or_create_client

from mpfs.common.util import from_json
from mpfs.core.zookeeper.shortcuts import get_checksum


@auth_check(role_names=[roles.ROLE_NAME_EDITOR, roles.ROLE_NAME_VIEWER, roles.ROLE_NAME_SUPERUSER])
def index(request):
    return render(request, 'common/index.html')


def ping(request):
    return HttpResponse('pong', status=200)


def who_am_i(request):
    return JsonResponse({
            'display_name': request.yauser.get_username(),
            'uid': request.yauser.uid,
            'environment': yenv.name
        },
        status=200
    )


@auth_check(role_names=[roles.ROLE_NAME_SUPERUSER])
def env(request):
    return JsonResponse({
        'zookeeper': {
            'env': settings.ZOOKEEPER_ENVIRONMENT,
            'root': settings.ZOOKEEPER_ROOT,
            'hosts': list(settings.ZOOKEEPER_HOSTS),
        },
        'node': {
            'hostname': socket.gethostname()
        },
        'settings': os.environ.get(
            'DJANGO_SETTINGS_MODULE'
        ).replace('settings', '').replace('_', '') or 'settings',
        'debug': settings.DEBUG,
        'django': {
            'database': {
                'default': {
                    'host': settings.DATABASES['default']['HOST'],
                    'name': settings.DATABASES['default']['NAME'],
                }
            }
        }
    })


@auth_check(role_names=[roles.ROLE_NAME_SUPERUSER])
def list_workers(request):
    zk_client = get_or_create_client()
    workers = zk_client.get_children(settings.ZOOKEEPER_WORKER_STATE_APPLIED_CONFIG_BASE_PATH)
    return JsonResponse(workers, safe=False)


@auth_check(role_names=[roles.ROLE_NAME_SUPERUSER])
def get_worker_state(request, identifier):
    zk_client = get_or_create_client()
    config_raw_data, stats = zk_client.get(
        os.path.join(settings.ZOOKEEPER_WORKER_STATE_APPLIED_CONFIG_BASE_PATH, identifier)
    )
    latest_raw_data, stats = zk_client.get(
        os.path.join(settings.ZOOKEEPER_WORKER_STATE_LATEST_CONFIG_BASE_PATH, identifier)
    )

    config_data = from_json(config_raw_data)
    latest_data = from_json(latest_raw_data)

    return JsonResponse({
        'config': {
            'data': config_data,
            'checksum': get_checksum(config_data)
        },
        'latest': {
            'data': latest_data,
            'checksum': get_checksum(latest_data)
        }
    })
