# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging
import posixpath

from django.conf import settings
from kazoo.client import KazooClient

from api_admin.signals import pre_config_changes, zk_client_created
from mpfs.common.util import from_json


logger = logging.getLogger('mpfs-api-admin.default')


client = None


def set_client(obj):
    """Установить глобальный клиент для работы с Зукипером.

    :param obj: Объект клиента, экземпляр класса `kazoo.KazooClient`.
    :return: Переданный объект клиента.
    """
    global client
    client = obj
    return client


def create_client():
    """Создает клиента для работы с Зукипером и устанавливает префиксный путь
    для всех последующих запросов.

    Создает префиксный путь для всех последующих запросов и устанавливает его.
    """

    # https://github.com/unbit/uwsgi/issues/1073#issuecomment-148641777
    # > According to https://uwsgi-docs.readthedocs.org/en/latest/Install.html
    # I should been able to run pip install uwsgi and import it into my python script,
    # but the uwsgi lib is never installed in site-packages.
    # > Hello, in that page there's no mention of uwsgi a generally available python
    # module. It's exported by the uwsgi process itself through the python plugin. So
    # it's available in your app but not on your shell.

    uwsgi = None
    try:
        import uwsgi
    except ImportError:
        pass

    hosts = ','.join(settings.ZOOKEEPER_HOSTS)
    if uwsgi:
        logger.info('Connecting to ZK hosts %s on worker %s.' % (settings.ZOOKEEPER_HOSTS, uwsgi.worker_id()))
    else:
        logger.info('Connecting to ZK hosts %s.' % settings.ZOOKEEPER_HOSTS)
    zk = ZookeeperClientWrapper(
        hosts=hosts,
        timeout=10,
        auth_data=[('sasl', '%s:%s' % (settings.ZOOKEEPER_USER, settings.ZOOKEEPER_PASSWORD))],
    )
    assert settings.ZOOKEEPER_ROOT.startswith(posixpath.sep)
    root = posixpath.join(settings.ZOOKEEPER_ROOT, settings.ZOOKEEPER_ENVIRONMENT)
    zk.start(timeout=10)
    zk.ensure_path(root)
    if uwsgi:
        logger.info('Changing ZK client root to %s on worker %s.' % (root, uwsgi.worker_id()))
    else:
        logger.info('Changing ZK client root to %s.' % root)
    zk.chroot = root
    zk_client_created.send(sender=None, zk=zk)
    return zk


def get_or_create_client():
    if client is not None:
        return client

    return set_client(create_client())


class ZookeeperClientWrapper(KazooClient):

    def set(self, path, new_value, user=None):
        """Сохранят предыдущую конфигурацию в базу перед изменениями"""
        old_value, stats = self.get(settings.ZOOKEEPER_PLATFORM_AUTH_SETTINGS_PATH)
        pre_config_changes.send(sender=None, user=user, config_dict=from_json(old_value))
        super(ZookeeperClientWrapper, self).set(path, new_value)
