#!/usr/bin/python
# -*- coding: utf-8 -*-
"""
https://jira.yandex-team.ru/browse/CHEMODAN-14192
"""
from itertools import ifilter

from mpfs.config import settings
from mpfs.analytics.common import BaseAnalyticsReportBuilder
from mpfs.common.util.database.connections import DirectRSConnectionFabric
from mpfs.core.user.constants import DEFAULT_SPACE_LIMIT, DEFAULT_LOCALE, SUPPORTED_LOCALES
from mpfs.common.static import tags
from mpfs.core.billing.product.catalog import CATALOG, PRODUCTS, PRIMARY_PRODUCTS
from mpfs.core.filesystem.quota import Quota


__all__ = ['UserDailyReportBuilder']


class UserDailyReportBuilder(BaseAnalyticsReportBuilder):

    SPACE_LIMIT_KEY = 'total_limit'
    SPACE_USED_KEY = 'used_space'
    RESOURCES_COUNT_KEY = 'resources_count'
    SPACE_BONUS_KEY = 'bonus_space'
    SPACE_BOUGHT_KEY = 'bought_space'
    SPACE_DISTRIBUTION_KEY = 'distribution_space'
    ACTIVATED_INVITES_KEY = 'user_activated_invites'
    SHARED_FOLDERS_COUNT_KEY = 'shared_folders_count'
    SHARED_FOLDERS_SIZE_KEY = 'shared_folders_size'
    SHARED_FOLDERS_INVITES_KEY = 'shared_folders_invites'
    SHARED_FOLDERS_ACTIVATED_INVITES_KEY = 'shared_folders_activated_invites'
    OTHER_SHARED_FOLDERS_SIZE_KEY = 'other_shared_folders_size'
    OTHER_SHARED_FOLDERS_COUNT_KEY = 'other_shared_folders_count'
    MAIL_ATTACHMENTS_SIZE_KEY = 'mail_attachments_size'
    LOCALE_KEY = 'locale'

    REPORT_FIELDS_DESCRIPTIONS = {
        SPACE_LIMIT_KEY: u'Общий лимит пользователя в диске.',
        SPACE_USED_KEY: u'Объём занятого места.',
        RESOURCES_COUNT_KEY: u'Количество файлов и папок на Диске всего.',
        SPACE_BONUS_KEY: u'Место, полученное в качестве бонуса.',
        SPACE_BOUGHT_KEY: u'Место, полученное за деньги.',
        SPACE_DISTRIBUTION_KEY: u'Место, полученное по дистрибуции.',
        ACTIVATED_INVITES_KEY: u"Список uid'ов, пришедших по инвайту uid'а.",
        SHARED_FOLDERS_COUNT_KEY: u'Общее число шареных папок пользователя.',
        SHARED_FOLDERS_SIZE_KEY: u'Общий объем шареных папок.',
        SHARED_FOLDERS_INVITES_KEY: u'Список уидов, которым уид выслал инвайт в общую папку.',
        SHARED_FOLDERS_ACTIVATED_INVITES_KEY: u'Список уидов, которые в этот день приняли инвайт уида в общую папку.',
        OTHER_SHARED_FOLDERS_SIZE_KEY: u'Объем шареных папок в которых он участвует.',
        OTHER_SHARED_FOLDERS_COUNT_KEY: u'Количество шареных папок в которых он участвует.',
        MAIL_ATTACHMENTS_SIZE_KEY: u'Объем файлов в аттаче Почты.',
        LOCALE_KEY: u'Локаль пользователя, указанная при регистрации в Диске.'
    }

    def __init__(self):
        self.collection_cache = {}
        self.connection_fabric = DirectRSConnectionFabric()
        self.quota = Quota()
        super(UserDailyReportBuilder, self).__init__()

    def get_report(self, source_data):
        self.collection_cache = {}
        return super(UserDailyReportBuilder, self).get_report(source_data)

    def get_collection(self, name, uid):
        if (name, uid) not in self.collection_cache:
            self.collection_cache[(name, uid)] = self.connection_fabric.get_collection_for_uid(name, uid)
        return self.collection_cache[(name, uid)]

    def collect_personal_quota(self, uid):
        """Расчитывает общий лимит диска, объем занятого места и количество файлов."""
        used = self.quota.used(uid=uid)
        limit = self.quota.limit(uid=uid)

        return {
            self.SPACE_LIMIT_KEY: limit,
            self.SPACE_USED_KEY: used,
        }

    def collect_personal_resources_count(self, uid):
        """Расчитывает количество файлов и папок пользователя."""
        collection = self.get_collection('user_data', uid)
        count = collection.find({'uid': uid}).count()
        return {self.RESOURCES_COUNT_KEY: count}

    def collect_personal_quota_composition(self, uid):
        """Расчитывает кол-во гигабайт, полученное в качестве бонуса, за деньги и по дистрибуции."""
        disk_info = self.get_collection('disk_info', uid)
        client_attributes = disk_info.find_one({'uid': uid, 'key': '/states/disk/billing'})
        client_attributes = dict(client_attributes) if client_attributes else {}
        market_code = client_attributes.get('market') or settings.billing['default_market']

        extract_pids = lambda products: [p[tags.billing.ID] for p in products]
        stat = {
            self.SPACE_BONUS_KEY: {
                'amount': DEFAULT_SPACE_LIMIT,
                'pids': extract_pids(CATALOG[market_code][tags.billing.BONUS]),
            },
            self.SPACE_BOUGHT_KEY: {
                'amount': 0,
                'pids': extract_pids(PRIMARY_PRODUCTS),
            },
            self.SPACE_DISTRIBUTION_KEY: {
                'amount': 0,
                'pids': extract_pids(CATALOG[market_code][tags.billing.DISTRIBUTION]),
            },
        }
        services = self.get_collection('billing_services', uid)
        billing_service_attributes = self.get_collection('billing_service_attributes', uid)
        for service in services.find({'uid': uid}):
            pid = service[tags.billing.PID]
            for k, v in stat.iteritems():
                if pid in v['pids']:
                    amount = PRODUCTS[pid][tags.billing.ATTRIBUTES].get(tags.billing.AMOUNT)
                    if PRODUCTS[pid].get(tags.billing.ALTATTRS):
                        sid = service['_id']
                        service_attrs = billing_service_attributes.find_one({'_id': sid})
                        if service_attrs:
                            amount = service_attrs.get(tags.billing.AMOUNT)
                    v['amount'] += amount

        ret = {}
        for k, v in stat.iteritems():
            ret[k] = v['amount']
        return ret

    def collect_personal_invites(self, uid):
        """Расчитывает список uid'ов, пришедших по инвайту uid'а."""
        collection = self.get_collection('invite_mpfs_referrals', uid)
        activated = [i.get('_id') for i in collection.find({'ref': uid})]
        return {self.ACTIVATED_INVITES_KEY: ','.join(activated)}

    def collect_personal_groups(self, uid):
        """Расчитывает список uid'ов, получивших и принявших инвайты в общую папку, число и общий размер общих папок uid'а."""
        invited_uids = set()
        activated_uids = set()
        user_share_size = 0
        groups_count = 0
        groups_collection = self.get_collection('groups', uid)
        # коллекции не шардированы, поэтому можно получить их до циклов по приглашенным пользователям
        group_invites_collection = self.get_collection('group_invites', uid)
        group_links_collection = self.get_collection('group_links', uid)
        for g in groups_collection.find({'owner': uid}):
            groups_count += 1
            user_share_size += g.get('size', 0)
            gid = g.get('_id')
            invited_uids.update(ifilter(None, [i.get('uid') for i in group_invites_collection.find({'gid': gid})]))
            activated_uids.update(ifilter(None, [i.get('uid') for i in group_links_collection.find({'gid': gid})]))
        return {
            self.SHARED_FOLDERS_COUNT_KEY: groups_count,
            self.SHARED_FOLDERS_SIZE_KEY: user_share_size,
            self.SHARED_FOLDERS_INVITES_KEY: ','.join(invited_uids),
            self.SHARED_FOLDERS_ACTIVATED_INVITES_KEY: ','.join(activated_uids),
        }

    def collect_personal_group_links(self, uid):
        """Расчитывает количесво и размер чужих папок, к которым пользователь имеет досутп."""
        group_links_collection = self.get_collection('group_links', uid)
        links_size = 0
        links_count = 0
        for l in group_links_collection.find({'uid': uid}):
            links_count += 1
            groups_collection = self.get_collection('groups', uid)
            group = groups_collection.find_one({'_id': l.get('gid')})
            if group:
                links_size += group.get('size')
        return {
            self.OTHER_SHARED_FOLDERS_SIZE_KEY: links_size,
            self.OTHER_SHARED_FOLDERS_COUNT_KEY: links_count,
        }

    def collect_personal_attachments(self, uid):
        """Расчитывает объём файлов в аттаче почты."""
        size = 0
        attach_data_collection = self.get_collection('attach_data', uid)
        result = attach_data_collection.aggregate([
            {'$match': {'uid': uid}},
            {'$group': {'_id': None, 'sum': {'$sum': '$data.size'}}}
        ])
        result = result.get('result')
        if result and len(result) > 0:
            size = result[0]['sum']
        return {self.MAIL_ATTACHMENTS_SIZE_KEY: size}

    def collect_userinfo(self, uid):
        """Собирает данные из user_info."""
        user_index_collection = self.get_collection('user_index', uid)
        user = user_index_collection.find_one({'_id': uid}, fields=['locale'])
        locale = None
        if user:
            locale = user.get('locale')
        if locale is None or locale not in SUPPORTED_LOCALES:
            locale = DEFAULT_LOCALE
        return {self.LOCALE_KEY: locale}
