# -*- coding: utf-8 -*-
from mpfs.common.errors import MPFSError, APIError


class PlatformError(MPFSError):
    description = ''

    def __init__(self, *args, **kwargs):
        http_response = self.response
        super(PlatformError, self).__init__(*args, **kwargs)
        self.response = http_response
        self.description = kwargs.get('description') or self.description


class PlatformBadRequest(PlatformError):
    response = 400
    description = 'bad request (missing arguments, body, etc)'


class PlatformFileCreationNotImplemented(PlatformBadRequest):
    description = 'unable to create file, look at GET /disk/resources/upload to create file'


class PlatformUnexpectedRequestBody(PlatformBadRequest):
    description = "handler doesn't expect any data in request body"


class PlatformValidationError(PlatformBadRequest):
    """Raised when data doesn't validate."""
    description = 'invalid data'


class PlatformMissedArguments(PlatformValidationError):
    description = 'some arguments are missing'


class PlatformMissedArgument(PlatformValidationError):
    description = 'some argument is missing'
    with_arg_template = '"%s" argument is missing'

    def __init__(self, description=None, arg=None, *args, **kwargs):
        self.arg = arg
        if self.arg and not description:
            description = self.with_arg_template % self.arg
        super(PlatformMissedArgument, self).__init__(description=description, *args, **kwargs)

class PlatformWrongTypeArguments(PlatformValidationError):
    description = 'wrong type for some arguments'


class PlatformRemoveNotEmptyFolderNonRecursively(PlatformBadRequest):
    description = 'unable to remove not empty folder non recursively'


class PlatformNotAuthorized(PlatformError):
    response = 401
    description = 'bad authorization'


#  HTTP 403


class PlatformForbidden(PlatformError):
    response = 403
    description = 'method not allowed (maybe you do not have allowed scopes)'


class PlatformPermissionDenied(PlatformForbidden):
    pass


# HTTP 404


class PlatformNotFound(PlatformError):
    response = 404
    description = 'resource not found'


#  HTTP 405


class PlatformNotAllowed(PlatformError):
    response = 405
    description = 'method not allowed (maybe you do not have allowed scopes)'


class PlatformNotImplemented(PlatformNotAllowed):
    description = 'this method is not implemented'


class PlatformUploadToExistentDirectory(PlatformNotAllowed):
    description = 'specified path points to existent directory'


class PlatformResourceAlreadyExists(PlatformNotAllowed):
    description = 'resource already exists'

# HTTP 409


class PlatformConflict(PlatformError):
    response = 409
    description = 'conflict'


class PlatformDirectoryDoesntExists(PlatformError):
    description = "specified directory doesn't exists"


# HTTP 412


class PlatformPreconditionFailed(PlatformError):
    response = 412
    description = 'precondition failed'


class PlatformFileAlreadyExists(PlatformPreconditionFailed):
    description = 'file already exists'


# HTTP 423


class PlatformResourceLocked(PlatformError):
    response = 423
    description = 'resource is locked'


# HTTP 503


class PlatformServiceUnavailable(PlatformError):
    response = 503
    description = 'service unavailable'


# HTTP 507
# This code is not part of RFC2616, it was introduced by WebDAV, but we have no better choice for that.
# However, any HTTP 1.1 client should be able to properly get and handle that code.


class PlatformInsufficientStorage(PlatformError):
    response = 507
    description = 'insufficient storage'


class PlatformOutOfStorageQuota(PlatformInsufficientStorage):
    description = 'you have exhausted storage quota'


class AssistantError(PlatformError):
    pass

class AssistantBadResponse(AssistantError, APIError):
    response = 503

class AssistantProfileNotFound(AssistantError):
    response = 404
    description = 'requested user is not found'

class AssistantUnknownProfileType(AssistantError):
    response = 501

class AssistantBadParameterFormat(AssistantError):
    response = 422
    
class AssistantParserError(AssistantError):
    response = 502

class MpfsProxyError(PlatformError):
    pass
    
class MpfsProxyBadResponse(MpfsProxyError, APIError):
    response = 503
    
class MpfsProxyNotFound(MpfsProxyBadResponse):
    response = 404
    
class MpfsProxyConflict(MpfsProxyBadResponse):
    response = 409
    
class MpfsProxyNoFreeSpace(MpfsProxyBadResponse):
    response = 497
    

class NotificationsError(PlatformError):
    pass

class NotificationsBadResponse(NotificationsError, APIError):
    response = 503

class MissedRequiredBodyFields(PlatformMissedArguments):
    pass

class MissedRequiredArguments(PlatformMissedArguments):
    pass

class MissedRequiredBodySubFields(PlatformMissedArguments):
    pass

class WrongTypeRequiredArguments(PlatformWrongTypeArguments):
    pass

class WrongTypeOptionalArguments(PlatformWrongTypeArguments):
    pass

class WrongTypeBody(PlatformWrongTypeArguments):
    pass

class WrongTypeRequiredBodyFields(PlatformWrongTypeArguments):
    pass

class WrongTypeOptionalBodyFields(PlatformWrongTypeArguments):
    pass

class WrongTypeBodySubFields(PlatformWrongTypeArguments):
    pass

class ExcessArguments(PlatformBadRequest):
    description = 'too many fields in body'
