# -*- coding: utf-8 -*-


import operator

from mpfs.common.util import natsort
from mpfs.common.errors import ResourceNotFound
from mpfs.common.forms import ResourceForm
from mpfs.common.forms import filters

import mpfs.engine.process

log = mpfs.engine.process.get_default_log()


class State(object):
    def __init__(self):
        self.folders_filtered = False
        self.files_filtered = False
        self.sort_applied = False
        self.offset_applied = False
        self.amount_applied = False
        self.order_applied = False


class FolderForm(ResourceForm):

    internal_filters = {
        'ctime' : filters.GapFilter,
        'mtime' : filters.GapFilter,
        'size'  : filters.GapFilter,
    }

    complex_sort_fields = ('media_type',)

    internal_bounds = {}

    def __init__(self, model):
        ResourceForm.__init__(self, model)
        self.child_folders = []
        self.child_files = []
        self._state = State()
        self._totally_filtered = 0
        self._chunk_folders_size = 0
        self._chunk_files_size = 0
        self.internal_bounds = (
            ('sort', self.sort),
            ('order', self.order),
            ('offset', self.cutoff),
            ('amount', self.cutoff),
        )

    @property
    def totally_filtered(self):
        if hasattr(self, 'model__num_res_filtered'):
            return self.model__num_res_filtered
        else:
            return self._totally_filtered or len(self.all_files())

    @property
    def chunk_numfiles(self):
        return self._chunk_files_size or len(self.files())

    @property
    def hasfolders(self):
        return int(
            bool(
                int(
                    self.model.hasfolders or
                    self.model.meta.get('hasfolders') or
                    len(self.model.children_items['folders'])
                )
            )
        )

    def folders(self):
        if (not self.child_folders and not self._state.folders_filtered
            and not self._state.offset_applied):
            for item in self.model.children_items['folders']:
                self.child_folders.append(item.form)
        return self.child_folders


    def files(self):
        if (not self.child_files and not self._state.files_filtered
            and not self._state.amount_applied):
            for item in self.model.children_items['files']:
                self.child_files.append(item.form)
        return self.child_files


    def all_folders(self, root=True, cut=False):
        result = []
        result.extend(self.folders())
        for fldr in self.folders():
            result.extend(fldr.all_folders(False))
        if root:
            result = self._manage_resources(result)
        if cut:
            if 'offset' in self.args['bounds']:
                result = result[self.args['bounds']['offset']:]
            if 'amount' in self.args['bounds']:
                result = result[:self.args['bounds']['amount']]
        self._chunk_folders_size = len(result)
        return result


    def all_files(self, root=True, cut=False):
        result = []
        result.extend(self.files())
        for fldr in self.folders():
            result.extend(fldr.all_files(False))
        if root:
            result = self._manage_resources(result)
        if cut:
            if 'offset' in self.args['bounds']:
                result = result[self.args['bounds']['offset']:]
            if 'amount' in self.args['bounds']:
                result = result[:self.args['bounds']['amount']]
        self._chunk_files_size = len(result)
        return result


    def _manage_resources(self, resources):
        if self.args['filter']:
            resources = self._filter(resources)
        if 'sort' in self.args['bounds']:
            sort_field = self.args['bounds']['sort']
            order = bool(self.args['bounds'].get('order', 1))
            resources = self._sort(resources, sort_field, order)
        return resources


    def update(self):
        pass


    def apply_filters(self):
        if self.args['filter']:
            self.child_folders = self._filter(self.folders())
            self.child_files = self._filter(self.files())
            self._state.folders_filtered = True
            self._state.files_filtered = True


    def apply_bounds(self):
        for bound_name, bound_handler in self.internal_bounds:
            if bound_name in self.args['bounds']:
                bound_handler()


    def order(self):
        if not self._state.order_applied:
            order = bool(self.args['bounds'].get('order'))
            if not order:
                self.child_folders = reversed(self.folders())
                self.child_files = reversed(self.files())
            self._state.order_applied = True


    def sort(self):
        if not self._state.sort_applied:
            sort_field = self.args['bounds']['sort']
            order = bool(self.args['bounds'].get('order', 1))
            files = self.files()
            folders = self.folders()
            self.child_folders = self._sort(folders, sort_field, order)
            self.child_files = self._sort(files, sort_field, order)
            self._state.sort_applied = True
            self._state.order_applied = True


    def cutoff(self):
        files = self.files()
        folders = self.folders()
        self._totally_filtered = self._totally_filtered or (len(files) + len(folders))
        if not self._state.offset_applied:
            offset = int(self.args['bounds'].get('offset', 0))
            if offset:
                files_offset = offset - len(folders)
                folders = folders[offset:]
                if files_offset and files_offset > 0:
                    files = files[files_offset:]
            self._state.offset_applied = True
        if not self._state.amount_applied:
            amount = self.args['bounds'].get('amount', None)
            if amount is not None:
                amount = int(amount)
                folders = folders[:amount]
                files_amount = amount - len(folders)
                if files_amount and files_amount > 0:
                    files = files[:files_amount]
                else:
                    files = []
            self._state.amount_applied = True
        self._chunk_files_size = len(files)
        self._chunk_folders_size = len(folders)
        self.child_files = files
        self.child_folders = folders

    def _sort(self, seq, s_field, order):
        if not seq:
            return []
        if s_field == 'name':
            result = sorted(seq, key=operator.attrgetter(s_field), reverse=not order)
        else:
            if s_field in self.complex_sort_fields:
                def _get_field(d):
                    return getattr(d, 'get_' + s_field)()
            else:
                def _get_field(d):
                    return getattr(d, s_field)
            try:
                result = sorted(seq, key=lambda d: (_get_field(d), d.name), reverse=not order)
            except AttributeError:
                noise = []
                tail = []
                for element in seq:
                    try:
                        _get_field(element)
                    except AttributeError:
                        tail.append(element)
                    else:
                        noise.append(element)
                noise.sort(key=lambda d: (_get_field(d), d.name), reverse=not order)
                tail.sort(natsort.natcasecmp, key=lambda d: d.name)
                result = noise + tail
        return result

    def _filter(self, seq):
        tmp_result = seq
        for fname, fval in self.args['filter'].iteritems():
            filter_mtd = self.internal_filters.get(fname, filters.BaseFilter)(fname, fval).check_form
            tmp_result = filter(filter_mtd, tmp_result)
        return tmp_result
