# -*- coding: utf-8 -*-
"""

MPFS
COMMON

Обшеиспользуемые методы и проч

"""
import StringIO
import calendar
import collections
import hashlib
import heapq
import logging
import os
import pstats
import re
import sys
import time
import threading
import traceback
import unicodedata
import urllib
import base64
import itertools

import demjson
import cjson
import ujson

from contextlib import contextmanager
from datetime import datetime
from functools import wraps

import mpfs.engine.process

from mpfs.common import errors


# Импортируем тут, чтоб импортировать отсюда в модулях, лежащих рядом с модулями collections.
# Кому интересно зачем всё так сложно: создайте в одном пакете модуль `collections.py` и `my.py`
# и попробуйте импортнуть в модуле `my.py` системный `defaultdict`. Удачи! :)
from collections import defaultdict, Counter


MAIL_RE = re.compile('([\w,\.\-\+]+)@([\w,\.\-]+\.\w+)$')
RE_ISO8601 = re.compile("^\d{4}-(\d{2})-(\d{2})T(\d{2}):(\d{2}):(\d{2})([-+]\d{2}:?\d{2}|Z)$")
RE_JPG = re.compile('^image/jpe?g$', flags=re.I)
RE_GPS = re.compile('(\d+) deg (\d+)\' (\d+\.\d+)" (\w)\, (\d+) deg (\d+)\' (\d+\.\d+)" (\w)')
HEM_MAP = {
    'N': 1,
    'S': -1,
    'E': 1,
    'W': -1,
}


YANDEX_MAILS = ('yandex.ru', 'ya.ru', 'yandex.by', 'narod.ru', 'yandex.com',
                'yandex.kz', 'yandex.ua', 'yandex.com.tr')

JSON_ENCODERS = {
    'ujson': ujson,
    'demjson': demjson,
    'cjson': cjson,
}


def get_first(iterable, default=None):
    """Получить первый элемент, а если его нет, то default"""
    return next(iter(iterable), default)


def time_to_str(timestamp):
    return datetime.fromtimestamp(int(timestamp)).strftime('%Y-%m-%d %H:%M:%S')


def iso8601(timestamp):
    offset = time.timezone / -(60*60)
    sign = '+' if offset > 0 else '-'
    return '%s%s%02d:00' % (datetime.fromtimestamp(int(timestamp)).strftime('%Y-%m-%dT%H:%M:%S'), sign, offset)


def match_iso8601(t):
    result = False
    matched = RE_ISO8601.match(t)
    if matched:
        m, d, H, M, S, _ = matched.groups()
        if 0 < int(m) <= 12 and \
            0 < int(d) <= 31 and \
            0 <= int(H) < 24 and \
            0 <= int(M) < 60 and \
            0 <= int(S) < 60:
            result = True
    return result


def timeit(func):
    """
    Декоратор. Измеряет время выполнения функции.

    Результат доступен в атрибуте этой же функции: `func.processing_time`
    """
    def wrapper(*args, **kwargs):
        start = time.time()
        res = func(*args, **kwargs)
        stop = time.time()
        wrapper.processing_time = stop - start
        return res
    wrapper.processing_time = 0.0
    return wrapper


def match_item(fltr, value):
    '''
    Проверка, попадает ли значение под фильтр
    Фильтр может быть сложным
    '''
    for k, v in fltr.iteritems():
        if k not in value:
            return False
        if isinstance(v, (list, tuple, set)):
            if value[k] not in v:
                return False
        elif isinstance(v, dict):
            return match_item(v, value[k])
        elif value[k] != v:
            return False
    return True


def email_valid(email):
    '''
    Проверка email на валидность
    '''
    try:
        if email and len(email) > 6:
            name, domain = email.split('@')
            email = name.encode('idna') + '@' + domain.encode('idna')

            import re
            prog = re.compile(
                r"(^[-!#$%&'*+/=?^_`{}|~0-9A-Z]+(\.[-!#$%&'*+/=?^_`{}|~0-9A-Z]+)*"  # dot-atom
                r'|^"([\001-\010\013\014\016-\037!#-\[\]-\177]|\\[\001-011\013\014\016-\177])*"'  # quoted-string
                r')@(?:[A-Z0-9](?:[A-Z0-9-]{0,61}[A-Z0-9])?\.)+(?:[A-Z]{2,6}\.?|[A-Z0-9-]{2,}\.?)$', re.IGNORECASE)  # domain
            if prog.match(email) != None:
                return True
        return False
    except Exception:
        return False


def format_dict_table(rows, column_names=None, max_column_width=None, border_style=2):
    """
    Returns a string representation of a tuple of dictionaries in a
    table format. This method can read the column names directly off the
    dictionary keys, but if a tuple of these keys is provided in the
    'column_names' variable, then the order of column_names will follow
    the order of the fields/keys in that variable.
    """
    if column_names or len(rows) > 0:
        lengths = {}
        rules = {}
        if column_names:
            column_list = column_names
            for row in rows:
                for k in column_list:
                    row[k] = row.get(k)
        else:
            try:
                column_list = rows[0].keys()
            except Exception:
                column_list = None
        if column_list:
            # characters that make up the table rules
            border_style = int(border_style)
            #border_style = 0
            if border_style >= 1:
                vertical_rule = ' | '
                horizontal_rule = '-'
                rule_junction = '-+-'
            else:
                vertical_rule = '  '
                horizontal_rule = ''
                rule_junction = ''
            if border_style >= 2:
                left_table_edge_rule = '| '
                right_table_edge_rule = ' |'
                left_table_edge_rule_junction = '+-'
                right_table_edge_rule_junction = '-+'
            else:
                left_table_edge_rule = ''
                right_table_edge_rule = ''
                left_table_edge_rule_junction = ''
                right_table_edge_rule_junction = ''

            if max_column_width:
                column_list = [c[:max_column_width] for c in column_list]
                trunc_rows = []
                for row in rows:
                    new_row = {}
                    for k in row.keys():
                        new_row[k[:max_column_width]] = string_converter(row[k])[:max_column_width]
                    trunc_rows.append(new_row)
                rows = trunc_rows

            for col in column_list:
                try:
                    rls = [len(string_converter(row[col])) for row in rows]
                except Exception:
                    rls = [len(string_converter(row[col])) for row in rows]
                lengths[col] = max(rls + [len(col)])
                rules[col] = horizontal_rule * lengths[col]

            template_elements = ["%%(%s)-%ss" % (col, lengths[col]) for col in column_list]
            row_template = vertical_rule.join(template_elements)
            border_template = rule_junction.join(template_elements)
            full_line = left_table_edge_rule_junction + (border_template % rules) + right_table_edge_rule_junction
            display = []
            if border_style > 0:
                display.append(full_line)
            display.append(left_table_edge_rule + (row_template % dict(zip(column_list, column_list))) + right_table_edge_rule)
            if border_style > 0:
                display.append(full_line)
            for row in rows:
                display.append(left_table_edge_rule + (row_template % row) + right_table_edge_rule)
            if border_style > 0:
                display.append(full_line)
            return "\n".join(display)
        else:
            return ''
    else:
        return ''


def human_size(size_bytes):
    """
    format a size in bytes into a 'human' file size, e.g. bytes, KB, MB, GB, TB, PB
    Note that bytes/KB will be reported in whole numbers but MB and above will have greater precision
    e.g. 1 byte, 43 bytes, 443 KB, 4.3 MB, 4.43 GB, etc
    """
    if size_bytes == 1:
        # because I really hate unnecessary plurals
        return "1 byte"

    suffixes_table = [('bytes', 0), ('KB', 0), ('MB', 1), ('GB', 2), ('TB', 2), ('PB', 2)]

    num = float(size_bytes)
    for suffix, precision in suffixes_table:
        if num < 1024.0:
            break
        num /= 1024.0

    if precision == 0:
        formatted_size = "%d" % num
    else:
        formatted_size = str(round(num, ndigits=precision))

    return "%s %s" % (formatted_size, suffix)


def string_converter(each):
    if isinstance(each, unicode):
        each = each.encode('utf-8')
    return str(each)


def time_parse(val):
    if not val or val == '0000-00-00 00:00:00':
        val = '0'
    elif not str(val).isdigit():
        RE_TIME = '(\d+)-(\d+)-(\d+).(\d+):(\d+):(\d+).*'
        try:
            _y, _mnth, _d, _h, _min, _s = re.search(RE_TIME, val).groups()
            val = int(time.mktime(datetime(int(_y), int(_mnth), int(_d), int(_h), int(_min), int(_s)).timetuple()))
        except Exception:
            val = calendar.timegm(time.strptime(val, '%d %b %Y %H:%M'))
    return int(val)


def timeout(func, args=(), kwargs={}, timeout_duration=1, default=None):
    class InterruptableThread(threading.Thread):
        def __init__(self):
            threading.Thread.__init__(self)
            self.exc = None
            self.result = None

        def run(self):
            try:
                self.result = func(*args, **kwargs)
            except Exception:
                self.exc = sys.exc_info()

    it = InterruptableThread()
    it.start()
    it.join(timeout_duration)
    if it.isAlive():
        it._Thread__stop()
        raise errors.MPFSTimeout()
    elif it.exc:
        e_type, e_value, e_stack = it.exc
        raise e_type, e_value, e_stack
    else:
        return it.result


def unstraighten_dictionary(var, delimiter='.'):
    """
    Обратная функция к straighten_dictionary
    """
    result = {}
    for key, value in var.iteritems():
        pointer = result
        parts = key.split(delimiter)
        for subkey in parts[:-1]:
            if subkey not in pointer:
                pointer[subkey] = {}
            pointer = pointer[subkey]
        pointer[parts[-1]] = value
    return result


def straighten_dictionary(var, prefix=None, delimeter='.'):
    '''
    На входе:
    {
        'a': {
            'b': 1
        },
        'y': 2,
        'b': {
            'x': {
                'c': 2, 'fsr': 2
            }
        }
    }

    На выходе:
    {
        'b.x.fsr': 2,
        'y': 2,
        'b.x.c': 2,
        'a.b': 1
    }
    '''
    result = {}
    for k, v in var.iteritems():
        if isinstance(v, dict):
            if prefix:
                key = '%s%s%s' % (prefix, delimeter, k)
            else:
                key = k
            result.update(straighten_dictionary(v, key, delimeter=delimeter))
        else:
            if prefix:
                new_value = {'%s%s%s' % (prefix, delimeter, k): v}
            else:
                new_value = {k: v}
            result.update(new_value)
    return result


def hashed(val):
    if not isinstance(val, (str, unicode)):
        val = str(val)
    elif isinstance(val, unicode):
        val = val.encode('utf-8')
    else:
        val = str(val)
    return hashlib.md5(val).hexdigest()


def normalized_hash(val):
    if val is not None:
        return str(val).lower()


def fulltime(offset=0.0):
    return str('%f' % (time.time() + offset)).replace('.', '')


def type_to_bool(val):
    try:
        return bool(int(val))
    except ValueError:
        return bool(val)


def wise_to_str(arg):
    if isinstance(arg, unicode):
        return arg
    try:
        return str(arg)
    except Exception:
        return str(arg.encode('utf-8'))


def normalize_unicode(s):
    if isinstance(s, unicode):
        try:
            unicode(s.encode('raw_unicode_escape'))
        except Exception:
            s = s.encode('raw_unicode_escape').decode('utf-8')
    elif isinstance(s, str):
        s = s.decode('utf-8')
    return s


def normalize_string(s):
    if isinstance(s, unicode):
        return unicodedata.normalize('NFC', s).encode('utf-8')
    else:
        return s

def name_from_path(path):
    path_split = filter(None, path.split('/'))
    if path_split:
        return path_split[-1]
    else:
        return '/'

def is_yandex_email(address):
    ''' DEPRECATED! ignores PDD users
    It returns login name or None, if is not from yandex'''
    result = MAIL_RE.match(address)
    if result and result.group(2) in YANDEX_MAILS:
        return result.group(1)
    else:
        return False


def decode_email(email):
    """Декодирует логин и домен из Punycode"""
    if email and '@' in email:
        login, domain = email.split('@', 1)
        return '%s@%s' % (login.decode('idna'), domain.decode('idna'))
    else:
        raise ValueError('No "@" in email')


def get_file_extension(path):
    return path.lower().split('.')[-1]


def get_default_json_encoder():
    from mpfs.config import settings
    default_json_encoder_name = settings.system['system']['default_json_encoder']
    default_json_encoder = JSON_ENCODERS.get(default_json_encoder_name, ujson)
    return default_json_encoder_name, default_json_encoder


def to_json(data, compactly=True, true_utf8=True, internal=False):
    """
    Преобразование python-структуры в JSON-строку

    internal - врубает принудительное использование ujson. Используем для взаимодействия внутренних компонент.
    """
    # https://st.yandex-team.ru/CHEMODAN-23166
    if internal:
        return ujson.encode(data, ensure_ascii=False)

    default_json_encoder_name, default_json_encoder = get_default_json_encoder()

    if default_json_encoder_name == 'demjson':
        return demjson.encode(data, compactly=compactly, encoding='utf-8')
    elif default_json_encoder_name == 'ujson':
        if true_utf8:
            return default_json_encoder.encode(data, ensure_ascii=False).replace('\/', '/').decode('utf-8')
        else:
            return default_json_encoder.encode(data).replace('\/', '/')
    else:
        return default_json_encoder.encode(data)


def from_json(text):
    """
    Преобразование JSON-строки в python-структуру

    """
    _, default_json_encoder = get_default_json_encoder()
    try:
        return default_json_encoder.decode(text)
    except Exception:
        return demjson.decode(text, encoding='utf-8')


class Singleton(object):
    def __new__(cls, *args, **kwargs):
        if not hasattr(cls, '_instance'):
            cls._instance = super(Singleton, cls).__new__(cls)
        return cls._instance


class Cached(object):
    _instances = {}

    @classmethod
    def reset(cls):
        cls._instances = {}


class CacheMixin(object):
    __cache = {}

    @classmethod
    def cache_reset(cls):
        cls.__cache = {}

    @classmethod
    def cache_set(cls, key, value):
        cls.__cache[key] = value

    @classmethod
    def cache_get(cls, key):
        return cls.__cache.get(key, None)

    @classmethod
    def cache_pop(cls, key):
        return cls.__cache.pop(key, None)


def ctimestamp():
    return int(time.time())


## {{{ http://code.activestate.com/recipes/577504/ (r3)
from sys import getsizeof, stderr
from collections import deque
try:
    from reprlib import repr
except ImportError:
    pass

def get_total_size(o, handlers={}):
    """ Returns the approximate memory footprint an object and all of its contents.

    Automatically finds the contents of the following builtin containers and
    their subclasses:  tuple, list, deque, dict, set and frozenset.
    To search other containers, add handlers to iterate over their contents:

        handlers = {SomeContainerClass: iter,
                    OtherContainerClass: OtherContainerClass.get_elements}

    """
    dict_handler = lambda d: itertools.chain.from_iterable(d.items())
    all_handlers = {tuple: iter,
                    list: iter,
                    deque: iter,
                    dict: dict_handler,
                    set: iter,
                    frozenset: iter,
                   }
    all_handlers.update(handlers)     # user handlers take precedence
    seen = set()                      # track which object id's have already been seen
    default_size = getsizeof(0)       # estimate sizeof object without __sizeof__

    def sizeof(o):
        if id(o) in seen:       # do not double count the same object
            return 0
        seen.add(id(o))
        s = getsizeof(o, default_size)

        for typ, handler in all_handlers.items():
            if isinstance(o, typ):
                s += sum(map(sizeof, handler(o)))
                break
        return s

    return sizeof(o)

def assert_no_exception(function, *args):
    try:
        function(*args)
    except Exception:
        return False
    else:
        return True

class method_caller(object):

    def __init__(self, method):
        self.method = method

    def __call__(self, *args, **kwargs):
        return self.method(*args, **kwargs)


def dict_keys_unicode_str(d):
    def convert(data):
        if isinstance(data, basestring):
            return str(data)
        elif isinstance(data, collections.Mapping):
            return dict(map(convert, data.iteritems()))
        elif isinstance(data, collections.Iterable):
            return type(data)(map(convert, data))
        else:
            return data
    return convert(d)


def iterfilelines(file_name):
    with open(file_name) as file_body:
        for line in file_body:
            line = line.strip()
            if line:
                yield line


def is_jpg(mimetype):
    return bool(RE_JPG.match(mimetype))


def unquote_qs_param(value):
    value = value.encode('utf-8')
    return urllib.unquote(value)


def convert_gps_deg_dec(c):
    deg_lat, min_lat, sec_lat, hem_lat, deg_lon, min_lon, sec_lon, hem_lon = RE_GPS.match(c).groups()
    dec_lat = int(deg_lat) + int(min_lat)/60.0 + float(sec_lat)/3600
    dec_lon = int(deg_lon) + int(min_lon)/60.0 + float(sec_lon)/3600
    result = '%s,%s' % (dec_lat*HEM_MAP[hem_lat], dec_lon*HEM_MAP[hem_lon])
    return result


def chunks(iterable, chunk_size=2):
    """
    Разбивает итерируемый объект на куски заданной длины.
    """
    for offset in xrange(0, len(iterable), chunk_size):
        yield iterable[offset:offset + chunk_size]


def chunks2(iterable, chunk_size=2):
    """Разбивает итерируемый объект на куски заданной длины.

    В отличие от `chunks` не требует от iterable реализацию __len__

    :rtype: generator[list]
    """
    chunk_size = int(chunk_size)
    if chunk_size <= 0:
        raise ValueError()

    chunk = []
    for item in iterable:
        chunk.append(item)
        if len(chunk) < chunk_size:
            continue
        yield chunk
        chunk = []
    if chunk:
        yield chunk


def grouped_chunks(iterable, key_getter, chunk_size=2):
    """
    Группирует элементы из итерируемого объекта по значению функции key_getter в блоки заданного размера

    :Example:
    >>> list(grouped_chunks(xrange(10), lambda x: x % 2, chunk_size=3))
    [(0, [0, 2, 4]), (1, [1, 3, 5]), (0, [6, 8]), (1, [7, 9])]
    >>> list(grouped_chunks(['aa', 'bbb', 'cc', 'dd', 'eee'], len))
    [(2, ['aa', 'cc']), (3, ['bbb', 'eee']), (2, ['dd'])]

    :param iterable: итерируемый объект
    :param key_getter: вызываемый объект, по возвращаемому значению которого должны быть сгруппированы элементы
    :param chunk_size: размер возвращаемых блоков
    :return: генератор блоков
    """
    chunk_map = defaultdict(list)
    for item in iterable:
        key = key_getter(item)
        chunk = chunk_map[key]
        chunk.append(item)
        if len(chunk) < chunk_size:
            continue
        yield key, chunk_map.pop(key)
    for key, chunk in chunk_map.iteritems():
        if chunk:
            yield key, chunk


def generete_name_with_suffix(name, suffix):
    """
    Добавит к имени суффикс с учетом расширения
    """
    path, ext = os.path.splitext(name)
    return path + suffix + ext


class AutoSuffixator(object):
    """
    Добавляет суффиксы к повторяющимся именам
    """
    def __init__(self):
        self.name_counter = Counter()

    def __call__(self, name):
        if self.name_counter[name] == 0:
            result = name
        else:
            suffix = ' (%i)' % self.name_counter[name]
            result = generete_name_with_suffix(name, suffix)
        self.name_counter[name] += 1
        return result


def say(message, level=logging.INFO, verbose=False, exc_info=False):
    """Утилита для вывода сообщения в лог и в консоль

    Для работы функции важно сначала вызвать функцию настраивающую логгеры,
    например setup_admin_script().
    Т.к. используется import, то функция медленная и не тред-сейф.

    :param any message: сообщение
    :param int level: уровень логирования
    :param bool verbose: выводить ли в консоль?
    :param bool exc_info: выводить ли трейсбек?
    :return:
    """

    logger = mpfs.engine.process.get_default_log()
    if not logger:
        raise RuntimeError('No logger found. Consider call setup() first')

    logger.log(level, message, exc_info=exc_info)

    if verbose:
        if isinstance(message, unicode):
            message = message.encode('utf-8')
        print message


@contextmanager
def trace_calls(cls, method_name):
    """
    Контекст-менеджер для сбора различной информации о вызове метода.

    Перехватывает число вызовов и вовзвращаемое значение.

    :type cls: type
    :type method_name: str

    :Example:
        >>> class A(object):
        ...     def f(self):
        ...         return 123
        >>> with trace_calls(A, 'f') as tracer:
        >>>     assert A().f() == 123
        >>>     assert tracer['total_calls'] == 1
        >>>     assert tracer['return_value'] == 123

    .. warning::
        **DEPRECATED:** Вместо этого метода используй :meth:`test.helpers.utils.catch_return_values`
    """
    counter = {
        'total_calls': 0,
        'return_value': None
    }
    method = getattr(cls, method_name)

    def count_method(*args, **kwargs):
        counter['total_calls'] += 1
        return_value = method(*args, **kwargs)
        counter['return_value'] = return_value
        return return_value

    try:
        setattr(cls, method_name, count_method)
        yield counter
    finally:
        setattr(cls, method_name, method)


class SuppressExceptions(object):
    """
    Замалчивает исключения и пишет их в лог.
    Может использоваться как декоратор или как контекст менеджер.

    Пример:
    >>> from mpfs.common.util import SuppressExceptions
    >>> @SuppressExceptions()
    ... def foo():
    ...     raise Exception()
    ...
    >>> foo()
    >>>
    >>>
    >>> def bar():
    ...     raise Exception()
    ...
    >>> with SuppressExceptions():
    ...     bar()
    ...
    >>>
    >>> @SuppressExceptions(default='123')
    ... def foo():
    ...     raise Exception()
    ...
    >>> foo()
    '123'
    >>>
    """
    def __init__(self, exceptions=None, default=None, **kwargs):
        """
        :param exceptions: Iterable of suppressing exceptions, if not provided then suppress all exceptions.
        :param log: Log to print traceback. Default is `mpfs.engine.process.get_error_log()`.
                    If specified but is None then doesn't print to log anything on exception.
        :param default: Default return value, if used as decorator and exception suppressed.
        """
        self.exceptions = exceptions or (Exception,)
        self.exceptions = tuple(self.exceptions)
        self.default = default
        self.log = kwargs.get('log', mpfs.engine.process.get_error_log())

    def __call__(self, f):
        """
        Decorator suppressing specified exceptions or all exceptions.

        :param f:
        :return: Actual decorator.
        """
        def wrapper(*args, **kwargs):
            result = self.default
            with self:
                result = f(*args, **kwargs)
            return result
        return wrapper

    def __enter__(self):
        """Метод необходимый для поддержки протокола контекст-менеджера."""
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        """
        Замалчивает и логирует или пробрасывает дальше по стеку исключение.

        Метод необходимый для поддержки протокола контекст-менеджера.
        """
        if exc_val is not None:
            if isinstance(exc_val, self.exceptions):
                if self.log is not None:
                    self.log.error(traceback.format_exc())
                return True
        return False


class Profiler(object):
    """
    Профилирует задекорированный или законтекстменеджереный код.
    Умеет выводить дамп cProfile в файл и статистику в лог.

    Может использоваться как декоратор или как контекст менеджер.
    """
    def __init__(self, snapshot_file=None, log=None):
        """
        :param snapshot_file: cProfile snapshot file.
        :param log: Лог в который выводится статистика.
        """
        self.snapshot_file = snapshot_file
        self.log = log

    def __call__(self, f):
        """
        Декоратор профилирующий метод

        :param f: Исходный метод.
        :return: Actual decorator.
        """
        def wrapper(*args, **kwargs):
            with self:
                result = f(*args, **kwargs)
            return result
        return wrapper

    def __enter__(self):
        """Врубает профилирование."""
        import cProfile
        self.profiler = cProfile.Profile()
        self.profiler.enable()
        return self.profiler

    def __exit__(self, exc_type, exc_val, exc_tb):
        """Вырубает профилирование и пишет дамп."""
        self.profiler.disable()

        if self.snapshot_file:
            self.profiler.dump_stats(self.snapshot_file)

        if self.log:
            s = StringIO.StringIO()
            sortby = 'time'
            ps = pstats.Stats(self.profiler, stream=s).sort_stats(sortby)
            ps.print_stats()
            self.log.info(s.getvalue())

        return False


class UnicodeBase64(object):
    """
    Обвязка над base64 для кодирования юникодовых строк

    На входе: basestring
    На выходе: unicode
    """
    @staticmethod
    def _conv_input_str(input_str):
        if isinstance(input_str, unicode):
            return input_str.encode('utf-8')
        return input_str

    @classmethod
    def _process(cls, code_func, s, *args, **kwargs):
        # base64 работает только c бинарными данными
        bin_s = cls._conv_input_str(s)
        return code_func(bin_s, *args, **kwargs).decode('utf-8')

    @classmethod
    def urlsafe_b64encode(cls, s):
        return cls._process(base64.urlsafe_b64encode, s)

    @classmethod
    def urlsafe_b64decode(cls, s):
        return cls._process(base64.urlsafe_b64decode, s)


def merge2(iterables, key=None, reverse=False):
    """
    Сливает несколько отсортированных источников данных в единый отсортированный.

    Аналог метода https://docs.python.org/3.5/library/heapq.html#heapq.merge.
    Поскольку мы используем python 2.7 в настоящее время, то мы не можем
    заиспользовать просто библиотечную функцию.

    :param iterables: Итерируемый объект итерируемых объектов.
    :param key: Callable объект, возвращающий ключ сравнения (пример lambda x: x.key).
    :param reverse: Сортировать ли в обратном порядке.
    """

    class CmpWrapper(object):
        """
        Вспомогательная обертка для работы с кучей.

        Поскольку библиотечная куча сейчас предоставляет открытый интерфейс только
        для работы с min-heap, то мы можем каждый объект обернуть в обертку
        которая >= меняет на <= при сравнении за счет домножения на -1. Таким
        образом самые большие объекты (но самые маленькие в обертке) окажутся на
        вершине кучи.
        """
        def __init__(self, _obj):
            self.obj = _obj

        def __cmp__(self, other):
            sign = 1
            if reverse:
                sign = -1

            if key:
                return sign * cmp(key(self.obj), key(other.obj))
            else:
                return sign * cmp(self.obj, other.obj)

    for (wrapped_obj, obj) in heapq.merge(*(((CmpWrapper(obj), obj) for obj in iterable) for iterable in iterables)):
        yield obj


def flatten_dict(d, parent_key=None, separator='.'):
    from mpfs.platform.utils import flatten_dict as platform_flatten_dict
    return platform_flatten_dict(d, parent_key=parent_key, separator=separator)


def _check_percentage_value(percentage):
    if not isinstance(percentage, int):
        raise TypeError('Percentage must be int but %s was given' % type(percentage))
    if percentage < 0 or percentage > 100:
        raise ValueError('Percentage value must be between 0 and 100')


def filter_uid_by_percentage(uid, percentage):
    """Возвращает True для <percentage> процентов юзеров"""
    if percentage == 100:
        return True
    if isinstance(uid, basestring) and not uid.isdigit():
        return False
    _check_percentage_value(percentage)
    return int(uid[-2:]) < percentage


def filter_value_by_percentage(value, percentage):
    """Возвращает True для <percentage> от хеша строки"""
    if percentage == 100:
        return True
    _check_percentage_value(percentage)
    # Берем хеш как сумму кодов символов в строковом представлении объекта
    try:
        str_value = str(value)
        hash_value = sum(ord(c) for c in str_value) % 100
    except UnicodeEncodeError:
        hash_value = 0
    return hash_value < percentage


def pairwise(iterable):
    "s -> (s0,s1), (s1,s2), (s2, s3), ..."
    a, b = itertools.tee(iterable)
    next(b, None)
    return itertools.izip(a, b)


def format_log_message(msg, **kwargs):
    kv = ['msg=' + msg] + ['%s=%s' % (k, v) for k, v in kwargs.iteritems()]
    return '; '.join(kv)


class SuppressAndLogExceptions(object):
    """Контекстный менеджер для пропуска указанной ошибки"""
    def __init__(self, logger, *exceptions):
        self.exceptions = exceptions
        self.logger = logger
        from mpfs.config import settings
        self.enabled = settings.feature_toggles['suppress_exception_block_enabled']

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_value, exc_traceback):
        if exc_type is None:
            return True
        if self.enabled and issubclass(exc_type, self.exceptions):
            self.logger.warning(traceback.format_exc())
            return True
        return False


def safe_convert(target_type, value, fallback=None):
    """
    Привести value к типу target_type, вернуть fallback если не удалось
    """
    try:
        return target_type(value)
    except ValueError:
        return fallback


def safe_to_int(value, fallback=None):
    return safe_convert(int, value, fallback=fallback)


class ThreadSafeCounter(object):
    def __init__(self):
        self._value = 0
        self._lock = threading.Lock()

    def increment(self, add_value=1):
        with self._lock:
            self._value += add_value
            return self._value

    def get_value(self):
        with self._lock:
            return self._value


def datetime_to_unixtime(value):
    """
    Преобразовать datetime в unixtime

    :param datetime value: значение для преобразования
    :return int: unixtime
    """
    return int(time.mktime(value.timetuple()))


def use_context(context):
    """
    Декоратор. Выполняет декорируемую функцию в переданном контексте.

    :param: объект поддерживающий context managment protocol
    :return: декорированная функция
    """
    def wrapper(fn):
        @wraps(fn)
        def wrapped(*args, **kwargs):
            with context:
                return fn(*args, **kwargs)
        return wrapped
    return wrapper


def with_retries(method, sleep_intervals, exception_class=Exception, on_exeption=None):
    """
    Перезапуск метода с заданными интервалами в случае его падения
    :param method: метод, который будет перезапускаться
    :param sleep_intervals: список с временными интервалами перезапусков
    :param exception_class: класс исключения, который надо отлавливать
    :return:
    """
    for sleep_seconds in sleep_intervals:
        try:
            return method()
        except exception_class:
            time.sleep(sleep_seconds)
            if on_exeption:
                on_exeption()
    return method()


def retry_decorator(exception, tries, delay=0):
    def retry(fun):
        def decorator(*args, **kwargs):
            return with_retries(lambda: fun(*args, **kwargs), [delay] * (tries - 1), exception)

        return decorator

    return retry
