# -*- coding: utf-8 -*-
import hashlib
import hmac
import re

from mpfs.common.util.urls import unquote, quote
from mpfs.config import settings
from ticket_parser2.api.v1 import UserTicket

CREDENTIAL_CRYPTER_SECRET = settings.credential_crypter['secret']
COOKIE = 'COOKIE'

AUTHORIZATION_HEADERS = ('AUTHORIZATION', 'YANDEX-AUTHORIZATION', 'X-FORWARDED-USER')
TVM_2_USER_TICKET_HEADER = 'X-YA-USER-TICKET'
HASH_THROUGH_HEADERS = {'TICKET', 'X-YA-SERVICE-TICKET'}
PUBLIC_PASSWORD_HEADER = 'DISK-PUBLIC-PASSWORD'

HTTP_SESSION_COOKIE_NAME = 'Session_id'
HTTPS_SESSION_COOKIE_NAME = 'sessionid2'

AUTHORIZATION_PATTERN = re.compile('(?P<name>oauth|xiva) (?P<credential>[^; $]*)', re.IGNORECASE)
OAUTH_QS_PATTERN = re.compile('([&?]oauth_token=[^&$]*)')
SESSION_ID_QS_PATTERN = re.compile('([&?]sessionid=[^&$]*)')
SSL_SESSION_ID_QS_PATTERN = re.compile('([&?]sslsessionid=[^&$]*)')


class CredentialSanitizer(object):
    """
    Класс, предназначенный для хеширования критичных для пользователей данных.

    Подобные данные, например, OAuth токены и куки, необходимо хэшировать, прежде чем они попадут в лог.
    """
    templates = {
        'oauth_token': (OAUTH_QS_PATTERN, 'oauth_token=%s'),
        'sessionid': (SESSION_ID_QS_PATTERN, 'sessionid=%s'),
        'sslsessionid': (SSL_SESSION_ID_QS_PATTERN, 'sslsessionid=%s'),
    }

    @classmethod
    def get_headers_list_with_sanitized_credentials(cls, raw_headers):
        """
        Хэширование креденшилов в хедерах

        :return [(<имя_заголовка>, <обработанное_значение>), ...]
        """
        hashed_headers_list = []
        for k, v in raw_headers.iteritems():
            new_v = v
            key_upper = k.upper()
            if key_upper in AUTHORIZATION_HEADERS:
                new_v = cls._hash_authorization_header_values(v)
            elif key_upper == COOKIE:
                new_v = cls._truncate_cookie_header_values(v)
            elif key_upper == TVM_2_USER_TICKET_HEADER:
                new_v = cls._sanitize_tvm_2_user_ticket(v)
            elif key_upper in HASH_THROUGH_HEADERS:
                new_v = cls.hash_credential(v)
            elif key_upper == PUBLIC_PASSWORD_HEADER:
                new_v = cls._sanitize_disk_public_password(v)
            hashed_headers_list.append((k, new_v))
        return hashed_headers_list

    @classmethod
    def sanitize_qs_parameter_in_url(cls, qs_name, url):
        if qs_name not in cls.templates:
            raise ValueError('Unsupported qs parameter to hash')
        pattern, replace_str = cls.templates[qs_name]

        found_substrings = pattern.findall(url)
        for i in found_substrings:
            credential_value_to_sanitize = i.split('=', 1)[1]
            prefix = i[0]  # необходимо искать с префиксом из-за коллизий в именах sessionid и sslsessionid
            if qs_name in {'sessionid', 'sslsessionid'}:
                sanitized_value = CredentialSanitizer._sanitize_session_id_value(unquote(credential_value_to_sanitize))
            else:
                sanitized_value = CredentialSanitizer.hash_credential(unquote(credential_value_to_sanitize))
            url = url.replace(i, (prefix + replace_str) % quote(sanitized_value))
        return url

    @staticmethod
    def hash_credential(value_to_hash):
        """Метод, предназначенный для хэширование всех попадающих в лог значений, требующих хэширования"""
        return hmac.new(CREDENTIAL_CRYPTER_SECRET, value_to_hash, hashlib.sha1).digest().encode('base64').rstrip('\n')

    @classmethod
    def _hash_authorization_header_values(cls, authorization):
        m = AUTHORIZATION_PATTERN.search(authorization)
        if not m:
            return authorization
        replace_values = {
            'name': m.groupdict()['name'],
            'credential': cls.hash_credential(m.groupdict()['credential']),
        }
        return AUTHORIZATION_PATTERN.sub('%(name)s %(credential)s' % replace_values, authorization)

    @classmethod
    def _truncate_cookie_header_values(cls, cookies):
        from mpfs.platform.utils import parse_cookie
        parsed_cookie = parse_cookie(cookies)
        for cookie_name in (HTTP_SESSION_COOKIE_NAME, HTTPS_SESSION_COOKIE_NAME):
            if cookie_name not in parsed_cookie:
                continue
            parsed_cookie[cookie_name] = cls._sanitize_session_id_value(parsed_cookie[cookie_name])

        return '; '.join('%s=%s' % (k, v) for k, v in parsed_cookie.iteritems())

    @staticmethod
    def _sanitize_session_id_value(session_id):
        return session_id.rsplit('.', 1)[0]

    @staticmethod
    def _sanitize_tvm_2_user_ticket(raw_user_ticket):
        return UserTicket.remove_signature(raw_user_ticket)

    @staticmethod
    def _sanitize_disk_public_password(password):
        return ''
