# -*- coding: utf-8 -*-
"""

MPFS
CORE

Шифрование AES

"""
import base64
import hashlib
import hmac

from Crypto import Random
from Crypto.Cipher import AES

from mpfs.common import errors

# Секретный ключ по умолчанию
DEFAULT_SECRET = 'f8dbcb3d33954d623c42373e0f09a349'

# Размер блока для AES шифрования, может быть 16 или 32
# Размер ключа должен совпадать с размером блока
DEFAULT_BLOCK_SIZE = 32

# Шифровальщик по умолчанию
DEFAULT_MODE = AES.MODE_ECB

# the character used for padding--with a block cipher such as AES, the value
# you encrypt must be a multiple of BLOCK_SIZE in length.  This character is
# used to ensure that your value is always a multiple of BLOCK_SIZE
PADDING = '{'

# one-liner to sufficiently pad the text to be encrypted
pad = lambda str, bsize: str + (bsize - len(str) % bsize) * PADDING


class CryptAgent(object):
    def __init__(self, key=DEFAULT_SECRET, bsize=DEFAULT_BLOCK_SIZE, mode=DEFAULT_MODE):
        self.key, self.bsize, self.mode = key, bsize, mode
        self.cipher = AES.new(key, mode)

    def encrypt(self, data, urlsafe=False, b64encoded=True):
        """Шифрование строки data
        """
        encode = base64.b64encode
        if urlsafe:
            encode = base64.urlsafe_b64encode

        result = self.cipher.encrypt(pad(data, self.bsize))
        if b64encoded:
            result = encode(result)

        return result

    def decrypt(self, hash, urlsafe=False, b64encoded=True):
        """Дешифровка хеша
        """
        decode = base64.b64decode
        if urlsafe:
            decode = base64.urlsafe_b64decode

        try:
            if b64encoded:
                hash = decode(hash)

            result = self.cipher.decrypt(hash).rstrip(PADDING)
        except Exception:
            raise errors.DecryptionError(hash)

        return result


class CryptAgentBase(object):
    def __init__(self, urlsafe=False):
        self._urlsafe = urlsafe

    @property
    def _b64encode(self):
        if self._urlsafe:
            return base64.urlsafe_b64encode
        else:
            return base64.b64encode

    @property
    def _b64decode(self):
        if self._urlsafe:
            return base64.urlsafe_b64decode
        else:
            return base64.b64decode

    def _pad(self, message, block_size):
        return pad(message, block_size)

    def _unpad(self, message):
        return message.rstrip(PADDING)


class AesCbcCryptAgent(CryptAgentBase):
    def __init__(self, key, urlsafe=False):
        super(AesCbcCryptAgent, self).__init__(urlsafe)
        self._key = key

    def encrypt(self, plaintext):
        """
        Шифрует сообщение. Возвращает base64 зашифрованного сообщения.

        :param plaintext: сообщение, которое надо зашифровать
        :return: base64 зашифрованного plaintext
        """
        if isinstance(plaintext, unicode):
            plaintext = plaintext.encode('utf-8')

        iv = Random.new().read(AES.block_size)
        aes = AES.new(self._key, AES.MODE_CBC, iv)
        plaintext = iv + plaintext

        return self._b64encode(aes.encrypt(self._pad(plaintext, AES.block_size)))

    def decrypt(self, ciphertext):
        """
        Расшифровывает сообщение. Принимает на вход base64 зашифрованного сообщения.

        :param ciphertext: base64 зашифрованного сообщения
        :return: расшифрованное сообщение
        """
        if isinstance(ciphertext, unicode):
            ciphertext = ciphertext.encode('ascii')

        ciphertext = self._b64decode(ciphertext)

        iv = ciphertext[:16]
        aes = AES.new(self._key, AES.MODE_CBC, iv)
        ciphertext = ciphertext[16:]

        result = self._unpad(aes.decrypt(ciphertext))
        return result


class HmacCryptAgent(CryptAgentBase):
    def __init__(self, key, digest_mode=hashlib.sha256, urlsafe=False):
        super(HmacCryptAgent, self).__init__(urlsafe)
        self._key = key
        self._digest_mode = digest_mode

    def sign(self, message):
        signature = hmac.new(key=self._key, msg=message, digestmod=self._digest_mode).digest()
        return self._b64encode(signature)
