# -*- coding: utf-8 -*-


import datetime

import mpfs.engine.process
import mpfs.common.util.dbnaming

from mpfs.config import settings
from mpfs.common.util.trees.avl import AVLTree, Node

SYSTEM_UID = mpfs.common.util.dbnaming.dbuid(settings.system['system']['uid'])


def put_mids_to_delete(mids_list, mids_collection, log, hid=None):
    """
    Помещение стидов файла без хардлинка на удаление
    """
    for mid in mids_list:
        log.info("\tput %s to %s" % (mid, mids_collection.name))
        data = {
            'date': datetime.datetime.now().strftime('%Y-%m-%d'),
        }
        if hid:
            data['hid'] = hid,
        mids_collection.put(SYSTEM_UID, mid, data, None)


class MongoShardDistributionNode(Node):
    def __init__(self, key, min_val, max_val):
        self.key = key
        self.parent = None
        self.leftChild = None
        self.rightChild = None
        self.height = 0
        self.min_val = min_val
        self.max_val = max_val

    def __str__(self):
        return str(self.key) + "(" + str(self.height) + ", " + str(self.balance())\
               + ") " + "[" + str(self.min_val) + ", " + str(self.max_val) + ")"


class MongoShardDistributionAVL(AVLTree):
    node_class = MongoShardDistributionNode

    def add_as_child(self, parent_node, child_node):
        node_to_rebalance = None
        if child_node.max_val < parent_node.min_val:
            if not parent_node.leftChild:
                parent_node.leftChild = child_node
                child_node.parent = parent_node
                if parent_node.height == 0:
                    node = parent_node
                    while node:
                        node.height = node.max_children_height() + 1
                        if not node.balance() in [-1, 0, 1]:
                            node_to_rebalance = node
                            break  # we need the one that is furthest from the root
                        node = node.parent
            else:
                self.add_as_child(parent_node.leftChild, child_node)
        elif child_node.min_val >= parent_node.max_val:
            if not parent_node.rightChild:
                parent_node.rightChild = child_node
                child_node.parent = parent_node
                if parent_node.height == 0:
                    node = parent_node
                    while node:
                        node.height = node.max_children_height() + 1
                        if not node.balance() in [-1, 0, 1]:
                            node_to_rebalance = node
                            break  # we need the one that is furthest from the root
                        node = node.parent
            else:
                self.add_as_child(parent_node.rightChild, child_node)
        else:
            return

        if node_to_rebalance:
            self.rebalance(node_to_rebalance)

    def find_in_subtree(self, node, value):
        if node is None:
            return None  # key not found
        if value < node.min_val:
            return self.find_in_subtree(node.leftChild, value)
        elif value >= node.max_val:
            return self.find_in_subtree(node.rightChild, value)
        else:  # key is equal to node key
            return node


def get_shard_bst(connection, db_name, collection_name):
    config_db = connection['config']
    tree = MongoShardDistributionAVL()
    for each in list(config_db['chunks'].find(
            {'ns': '%s.%s' % (db_name, collection_name)})):
        min_value = each['min'].values()[0]
        if isinstance(min_value, dict):
            min_value = None
        max_value = each['max'].values()[0]
        if isinstance(max_value, dict):
            max_value = None
        tree.insert(each['shard'], min_value, max_value)
    return tree
