# -*- coding: utf-8 -*-
"""
Порядок добавления условия включения эксперимента:
  1. Написать класс для проверки на включение эксперимента
  2. Добавить класс в ExperimentClauseFacade.available_clauses_classes
"""
import traceback
import hashlib

from mpfs.core.services.uaas_service import new_uaas
from mpfs.engine.process import get_error_log


error_log = get_error_log()


class ExperimentClause(object):
    CONFIG_KEY = None

    @classmethod
    def build_experiment(cls, settings, experiment_name):
        """Создать условие включения эксперимента.

        :param settings: настройки включения эксперимента
        :param experiment_name: название эксперимента
        :return:
        """
        raise NotImplementedError()

    @classmethod
    def is_proper_class(cls, config_key):
        return cls.CONFIG_KEY == config_key

    def is_active(self, context):
        """Проверить включен ли эксперимент для указанного контекста.

        :param context:
        :return:
        """
        raise NotImplementedError()


class ExperimentByUidClause(ExperimentClause):
    """
    by_uid:
      enabled_for: ['12345']
      enabled_for_percentage: 57
    """
    CONFIG_KEY = 'by_uid'
    CHARS_NUM_TO_USE = 6  # выбрано экспериментально по времени и точности
    MAX_NUM = float(16 ** CHARS_NUM_TO_USE)

    def __init__(self, enabled_uids_surely=tuple(), percentage=0, seed=None):
        self.enabled_uids_surely = enabled_uids_surely
        self.percentage = percentage
        self.seed = seed

    @classmethod
    def build_experiment(cls, settings, experiment_name):
        experiment = cls(
            enabled_uids_surely=settings.get('enabled_for', set()),
            percentage=settings.get('enabled_for_percentage', 0),
            seed=experiment_name
        )
        return experiment

    def is_active_by_uid(self, uid):
        if uid is None:
            return False

        if uid in self.enabled_uids_surely:
            return True

        if self.percentage == 100:
            return True
        if self.percentage == 0:
            return False

        hashed_uid = hashlib.md5(str(uid) + str(self.seed)).hexdigest()
        return int(hashed_uid[-self.CHARS_NUM_TO_USE:], 16) / self.MAX_NUM * 100 < self.percentage

    def is_active(self, context):
        uid = context.get('uid')
        return uid and self.is_active_by_uid(uid)


class ExperimentByUaaSClause(ExperimentClause):
    """
    by_uaas:
      flag: disk_pro2019_exp
    """
    CONFIG_KEY = 'by_uaas'

    def __init__(self, flag):
        self.flag = flag

    @classmethod
    def build_experiment(cls, settings, experiment_name):
        experiment = cls(flag=settings.get('flag', ''))
        return experiment

    def is_active_by_uid(self, uid):
        try:
            experiments = new_uaas.get_disk_experiments(uid=uid)
        except Exception:
            error_log.error("failed to get experiments for: %s" % uid)
            error_log.error(traceback.format_exc())
            return False

        return any(self.flag in experiment.get('CONTEXT', {}).get('DISK', {}).get('flags', [])
                   for experiment in experiments)

    def is_active(self, context):
        uid = context.get('uid')
        return uid and self.is_active_by_uid(uid)


class ExperimentClauseFacade(object):
    available_clauses_classes = [
        ExperimentByUidClause,
        ExperimentByUaaSClause,
    ]

    @staticmethod
    def build_clause(name, clause_type, clause_settings):
        for clause_cls in ExperimentClauseFacade.available_clauses_classes:
            if clause_cls.is_proper_class(clause_type):
                return clause_cls.build_experiment(clause_settings, experiment_name=name)
        else:
            raise NotImplementedError('Unexpected experiment type %s' % clause_type)

    @staticmethod
    def is_clause_valid(experiment_clause, context):
        for clause_cls in ExperimentClauseFacade.available_clauses_classes:
            if type(experiment_clause) == clause_cls:
                return experiment_clause.is_active(context)
        return False
