# -*- coding: utf-8 -*-
"""
Порядок добавления исключения:
  1. Написать класс для проверки на исключение
  2. Добавить класс в ExperimentClauseFacade.available_exclusions_clauses_classes
"""
import traceback
import re

from mpfs.engine.process import get_error_log


error_log = get_error_log()


class ExperimentExclusionClause(object):
    CONFIG_KEY = None

    @classmethod
    def build_experiment(cls, settings):
        """Создать условие исключения из эксперимента.

        :param settings: настройки исключения из эксперимента
        :return:
        """
        raise NotImplementedError()

    @classmethod
    def is_proper_class(cls, config_key):
        return cls.CONFIG_KEY == config_key

    def is_excluded(self, context):
        """Проверить исключен ли из эксперимента указанный контекст.

        :param context:
        :return:
        """
        raise NotImplementedError()




class ExperimentByUidExclusionClause(ExperimentExclusionClause):
    """
    by_uid:
      disabled_for: ['23985']
      login_regex: ['^yndx-ufo-test-\d+$']
    """
    CONFIG_KEY = 'by_uid'

    def __init__(self, disabled_uids_surely=tuple(), login_regex=tuple()):
        self.disabled_uids_surely = disabled_uids_surely
        self.logins_regex = login_regex

    @classmethod
    def build_experiment(cls, settings):
        experiment = cls(
            disabled_uids_surely=settings.get('disabled_for', set()),
            login_regex=[re.compile(pattern)
                         for pattern in settings.get('login_regex', set())]
        )
        return experiment

    def is_excluded_by_login_pattern(self, uid):
        if not self.logins_regex:
            return False

        from mpfs.core.services.passport_service import passport
        try:
            userinfo = passport.userinfo(uid)
        except Exception:
            error_log.error("failed to check experiments exclusions for: %s" % uid)
            error_log.error(traceback.format_exc())
            return False

        login = userinfo.get('login', '')
        for regex in self.logins_regex:
            if regex.match(login):
                return True

        return False

    def is_excluded_by_uid(self, uid):
        if uid is None:
            return False

        if uid in self.disabled_uids_surely:
            return True

        if self.is_excluded_by_login_pattern(uid):
            return True

        return False

    def is_excluded(self, context):
        uid = context.get('uid')
        return uid and self.is_excluded_by_uid(uid)


class ExperimentExclusionClauseFacade(object):
    available_exclusion_clauses_classes = [
        ExperimentByUidExclusionClause,
    ]

    @staticmethod
    def build_clause(name, clause_type, clause_settings):
        for clause_cls in ExperimentExclusionClauseFacade.available_exclusion_clauses_classes:
            if clause_cls.is_proper_class(clause_type):
                return clause_cls.build_experiment(clause_settings)
        else:
            raise NotImplementedError('Unexpected experiment type %s' % clause_type)

    @staticmethod
    def is_clause_valid(experiment_clause, context):
        for clause_cls in ExperimentExclusionClauseFacade.available_exclusion_clauses_classes:
            if type(experiment_clause) == clause_cls:
                return experiment_clause.is_excluded(context)
        return False
