# -*- coding: utf-8 -*-
"""
Набор инструментов для работы с экспериментами.

https://wiki.yandex-team.ru/disk/mpfs/meps/mep-044/

Позволяет не следить за тем чтобы передать нужные данные для проверки включенности эксперимента:
  * выставляет контекст (uid) в едином месте

"""
import copy

from contextlib import contextmanager
from functools import wraps

from mpfs.common.util.experiments.clauses import ExperimentClauseFacade, ExperimentByUidClause
from mpfs.common.util.experiments.exclusion_clauses import ExperimentExclusionClauseFacade
from mpfs.config import settings


class ExperimentManager(object):
    experiments_by_key = None  # type: dict[str, Experiment]
    context = None

    def __init__(self):
        self.experiments_by_key = {}
        self.context = {}
        self.load_experiments_from_conf()

    def load_experiments_from_conf(self):
        for exp_name, exp_description in settings.experiments.iteritems():
            experiment_enabled = exp_description['enabled']
            if not isinstance(experiment_enabled, bool):
                raise KeyError('Every experiment must have `enabled` field and be of bool type')

            new_experiment = Experiment(experiment_enabled)
            for clause_type, clause_settings in exp_description.get('exclusions', {}).iteritems():
                new_experiment.add_exclusion(ExperimentExclusionClauseFacade.build_clause(
                    exp_name, clause_type, clause_settings))
            for clause_type, clause_settings in exp_description['clauses'].iteritems():
                new_experiment.add_clause(ExperimentClauseFacade.build_clause(
                    exp_name, clause_type, clause_settings))
            self.experiments_by_key[exp_name] = new_experiment

    def is_feature_active(self, experiment_key):
        experiment = self.experiments_by_key.get(experiment_key)
        if experiment is None:
            raise NotImplementedError()
        return experiment.is_active(self.context)

    def update_context(self, **kw):
        self.context.update(kw)


class Experiment(object):
    """
    Объект представляет некоторый эксперимент

    Если у эксперимента несколько условий, то эксперимент активен, когда все условия удовлетварительны
    """
    enabled = False
    experiment_clause_list = None
    experiment_exclusions_list = None

    def __init__(self, enabled):
        self.enabled = enabled
        self.experiment_clause_list = []
        self.experiment_exclusions_list = []

    def add_clause(self, experiment):
        self.experiment_clause_list.append(experiment)

    def add_exclusion(self, experiment):
        self.experiment_exclusions_list.append(experiment)

    def is_active(self, context):
        if not self.enabled:
            return False

        for experiment_exclusion in self.experiment_exclusions_list:
            if ExperimentExclusionClauseFacade.is_clause_valid(experiment_exclusion, context):
                return False

        for experiment_clause in self.experiment_clause_list:
            if ExperimentClauseFacade.is_clause_valid(experiment_clause, context):
                return True

        return False


@contextmanager
def change_experiment_context_with(**kwargs):
    old_context_values = {}
    try:
        for k in kwargs.iterkeys():
            old_context_values[k] = experiment_manager.context.get(k, None)
        experiment_manager.update_context(**kwargs)
        yield
    finally:
        experiment_manager.update_context(**old_context_values)


@contextmanager
def enable_experiment_for_uid(experiment_key, uid, enabled=True):
    """
    Контекстный менеджер ТОЛЬКО ДЛЯ ТЕСТОВ
    """
    experiment = experiment_manager.experiments_by_key[experiment_key]

    try:
        mocked_experiment = copy.deepcopy(experiment)
        mocked_experiment.add_clause(ExperimentByUidClause([uid], 0, None))
        mocked_experiment.enabled = enabled

        experiment_manager.experiments_by_key[experiment_key] = mocked_experiment
        yield
    finally:
        experiment_manager.experiments_by_key[experiment_key] = experiment


def experiment_gateway_wrapper(experiment_key, return_value=None):
    """Декоратор для подавления логики экспериментов.

    :param experiment_key: имя эксперимента
    :param return_value: значение, которое следует вернуть, если эксперимент отключен
    """

    def func_decorator(f):

        @wraps(f)
        def wrapped_func(*args, **kwargs):
            if not experiment_manager.is_feature_active(experiment_key):
                return return_value
            return f(*args, **kwargs)

        return wrapped_func

    return func_decorator


experiment_manager = ExperimentManager()
