# -*- coding: utf-8 -*-
import os
import sys
import mimetypes

import mpfs.engine.process

from mpfs.config import settings

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


DEFAULT_MIME_TYPE = settings.services['zaberun']['default_content_type']


class MediaType(object):
    AUDIO = 'audio'
    BACKUP = 'backup'
    COMPRESSED = 'compressed'
    DATA = 'data'
    DEVELOPMENT = 'development'
    DISK_IMAGE = 'diskimage'
    DOCUMENT = 'document'
    ENCODED = 'encoded'
    FONT = 'font'
    IMAGE = 'image'
    SETTINGS = 'settings'
    TEXT = 'text'
    VIDEO = 'video'
    WEB = 'web'
    EXECUTABLE = 'executable'
    SPREADSHEET = 'spreadsheet'
    FLASH = 'flash'
    BOOK = 'book'
    UNKNOWN = 'unknown'

# Не меняйте цифры в builtin_extensions! НИКОГДА!
builtin_extensions = {
    MediaType.AUDIO:        0,
    MediaType.BACKUP:       1,
    MediaType.COMPRESSED:   2,
    MediaType.DATA:         3,
    MediaType.DEVELOPMENT:  4,
    MediaType.DISK_IMAGE:   5,
    MediaType.DOCUMENT:     6,
    MediaType.ENCODED:      7,
    MediaType.FONT:         8,
    MediaType.IMAGE:        9,
    MediaType.SETTINGS:     10,
    MediaType.TEXT:         11,
    MediaType.VIDEO:        12,
    MediaType.WEB:          13,
    MediaType.EXECUTABLE:   14,
    MediaType.SPREADSHEET:  15,
    MediaType.FLASH:        16,
    MediaType.BOOK:         17,
    MediaType.UNKNOWN:      18,
}

extensions = {}

for name in builtin_extensions.keys():
    name = "mpfs.common.util.extlib." + name
    __import__(name)
    m = sys.modules[name]
    extensions.update(m.extensions)


def getGroupNumber(name):
    return builtin_extensions.get(name)


def getGroupByNumber(number):
    for k, v in builtin_extensions.iteritems():
        if v == number:
            return k


def getGroupOnlyByName(name):
    name = name.lower()
    if '.' not in name:
        return

    ext = name.split('.')[-1]

    return getByExtension(ext, only_common=True)


def getExtensionsFromGroup(name):
    return extensions[name].keys()


def getMimeType(name):
    ext = '.' + name.split('.')[-1:][0].lower()
    result = mimetypes.types_map.get(ext)
    if result == 'application/octet-stream':
        result = None
    return result


def getMimeTypesFromGroup(name):
    exts = getExtensionsFromGroup(name)
    return [v for k,v in mimetypes.types_map.iteritems() if k[1:] in exts and v != 'application/octet-stream']


def getGroupOnlyByMimetype(mtype, name=None):
    _map = {}
    result = ''
    if name:
        ext = name.split('.')[-1:][0].lower()
        expected = getExtensionsForMime(mtype)
        if ext in expected:
            expected = set([ext])
    else:
        expected = getExtensionsForMime(mtype)

    if mtype:
        for group, values in extensions.iteritems():
            coincidents = len(filter(lambda x: x in values, expected))
            if coincidents:
                _map[coincidents] = group
        if expected and _map:
            result = _map.get(max(_map))
    return result


def getExtensionsForMime(mime):
    mime = mime or ''
    return set(x[1:] for x in mimetypes.guess_all_extensions(mime))


def getGroupByMimetype(mtype, name):
    return getGroupOnlyByMimetype(mtype, name) or getGroupOnlyByName(name) or 'unknown'


def getGroupByName(name, mtype=''):
    return getGroupOnlyByName(name) or getGroupOnlyByMimetype(mtype) or 'unknown'


def getMimeByName(name):
    result = ''
    if name:
        ext = name.split('.')[-1:][0].lower()
        result = (getGroup(ext) or 'unknown') + '/' + ext
    return result


def getByExtension(extension, group=None, only_common=False):
    _cmmn = []
    _uncmmn = []
    if group and group in extensions:
        if extension in extensions[group]:
            if only_common and extensions[group][extension][0] == 0: return None
            return group
    else:
        for group in extensions:
            if extension in extensions[group]:
                if only_common and extensions[group][extension][0] == 0:
                    _uncmmn.append(group)
                else:
                    return group
    if _uncmmn:
        return _uncmmn[0]


def getMediaTypeByName(name):
    if name:
        ext = name.rsplit('/', 1)[-1].rsplit('.')[-1:][0].lower()
        return getByExtension(ext)


def getByMimeType(mimetype, group=None, only_common=False):
    result = []
    import mimetypes
    for extension in mimetypes.guess_all_extensions(mimetype):
        result.extend(getByExtension(extension[1:]))
    return result


def getByDescription(description, only_common=False):
    result = []
    description = description.lower()
    for group in extensions:
        for extension in extensions[group]:
            if only_common and extensions[group][extension][0] == 0: continue
            if description in extensions[group][extension][1].lower():
                result.append(extension)
    return result


def getGroup(extension):
    result = filter(lambda (group, exts): extension in exts, extensions.iteritems())
    if result:
        return result[0][0]


if __name__ == '__main__':
    import unittest

    class TestCase(unittest.TestCase):
        def test_by_extension1(self):
            test = getByExtension('001')
            self.assertEqual(test, ['HJSplit Split Archive'])
        def test_by_extension2(self):
            test = getByExtension('wav', group='audio')
            self.assertEqual(test, "Windows WAVE Sound File")
        def test_by_description1(self):
            test = getByDescription('PHP')
            self.assertEqual(sorted(test), sorted(['phtm', 'phtml', 'php3', 'php4', 'php5']))
        def test_by_description2(self):
            test = getByDescription('Hypertext', only_common=True)
            self.assertEqual(sorted(test), sorted(['htm', 'html', 'php', 'xhtml']))
        def test_by_mimetype(self):
            test = getByMimeType('application/x-shockwave-flash')
            self.assertEqual(test, ['Macromedia Flash Movie'])
        def test_group(self):
            test = getGroup('zip')
            self.assertEqual(test, 'compressed')

    suite = unittest.TestLoader().loadTestsFromTestCase(TestCase)
    unittest.TextTestRunner(verbosity=3).run(suite)
