# -*- coding: utf-8 -*-
"""
Для работы с IP адрессами
"""

import socket
import struct


def ip_to_long(ip):
    """Преобразует IP адрес в числовое представление.

    Поддреживает IPv4 и IPv6.

    :param ip: строкое представление IP адреса

    :returns: числовое представление IP адреса
    """
    # Сначала пытается преобразовать адресс как IPv4
    try:
        packed_ip = socket.inet_pton(socket.AF_INET, ip)
        result = struct.unpack('!L', packed_ip)[0]
    except socket.error:
        # Если не получилось, пытаеся преобразовать как IPv6
        packed_ip = socket.inet_pton(socket.AF_INET6, ip)
        # Распаковка:
        # IPv6 адрес сохраняется в виде массива беззнаковых чисел (и сам адрес является беззнаковым). Суммарно 128 бит.
        # Для простоты будем распаковывать половинами:
        # addr: <unsigned long long><unsigned long long>
        # Получив пару 64-битовых беззнаковых чисел, получаем итоговый результат путем выставления битов каждой
        # половины (на примере 8-битовых):
        # addr =
        #        AAAAAAAA........ <high part> << <bits count in parts>
        #       |
        #        ........BBBBBBBB <low part>
        #        AAAAAAAABBBBBBBB
        # Примечание: можно было бы распаковывать и более мелкими частями, например по 32 бита:
        # add: <unsigned long><unsigned long><unsigned long><unsigned long>
        # но в этом случае пришлось бы делать больше операций
        # Детали:
        # 1. про хранение в памяти:
        # http://man7.org/linux/man-pages/man7/ipv6.7.html
        # 2. про распаковку:
        # https://docs.python.org/2/library/struct.html#format-characters
        high, low = struct.unpack('!QQ', packed_ip)
        result = (high << 64) | low
    return result


def _is_network_address(address):
    """Проверяет является ли адрес - адресом сети."""
    # TODO(kis8ya): переделать на нормальную работу с адресами сетей
    return address.find('/') != -1


def _network_address_to_block(address):
    # default: IPv4
    bits_count = 32
    if address.find(':') != -1:
        # IPv6
        bits_count = 128

    ip, mask_value = address.split('/')
    mask_value = int(mask_value)
    ip = ip_to_long(ip)

    shift = bits_count - mask_value
    network = ip >> shift << shift
    broadcast = network | (1 << shift) - 1

    return network, broadcast


class IPRange(object):
    """
    Диапазон IP адресов.

    Предоставляет проверку вхождения адреса в заданный диапазон адресов.
    Позволяет задавать диапазон подсетью.

    """
    def __init__ (self, start, end=None):
        """
        :param start: начальный IP адрес ("192.168.0.19") или подсеть ("192.168.0.0/24")
        :type start: str
        :param end: конец диапазона IP адресов (можно указывать только если `start` не был задан как подсеть)
        :type end: str
        """
        if _is_network_address(start):
            start, end = _network_address_to_block(start)
        else:
            if start.find('-') != -1:
                # start в формате диапазона:
                #   "10.0.0.1 - 10.0.0.19"
                start, end = start.split('-')
                start = start.strip()
                end = end.strip()
            elif end is None:
                # диапазон из одного адреса
                end = start

            start = ip_to_long(start)
            end = ip_to_long(end)

        self.start = min(start, end)
        self.end = max(start, end)

    def __contains__ (self, ip):
        """Проверяет входит ли адрес в диапазон."""
        if isinstance(ip, basestring):
            ip = ip_to_long(ip)
        else:
            raise TypeError('IP address should be basestring: %s' % ip)

        return self.start <= ip <= self.end


class IPRangeList(object):
    """Список диапазонов IP адресов."""
    def __init__ (self, *args):
        self.ranges = tuple(map(IPRange, args))

    def __contains__ (self, ip):
        """Проверяет входит ли IP адрес в список диапазонов адресов."""
        for ip_range in self.ranges:
            if ip in ip_range:
                return True
        return False

    def __iter__ (self):
        """Возврает итератор по диапазонам адресов из списка."""
        for ip_range in self.ranges:
            yield ip_range
