# -*- coding: utf-8 -*-
import Queue
import multiprocessing
import os
import time
import traceback
import shelve

from anydbm import error as AnyDbmError

import mpfs.engine.process
from mpfs.common.util import logger


TERMINAL_OBJECT = '9b11a0f5-778f-4cf6-91ac-8b17ca8c3cf7'


class BaseTaskQueue(object):
    def __init__(self, target, max_processes=None, callback=None, wait_time=0.1, error_handler=None, log=None,
                 termination_timeout=None):
        """
        :param target: Функция которую очередь будет применять к исходным данным. Принимает объект для обработки.
        :param max_processes: Максимальное количество процессов используемых очередью.
        :param callback: Метод который получит результат выполнения target.
        :param wait_time: Время ожидания между попытками запустить новую задачу.
        :param error_handler: Функция которая будет вызвана в случае ошибки выполнения target или callback.
                              Принимает 2 аргумента: объект на котором произошла ошибка, выброшенное исключение.
        :param termination_timeout: Время ожидания корректного завершения подпроцессов при завершении обработки очереди.
                                    Если значение None, то таймаута нет, процесс будет ожидать завершения подпроцессов
                                    столько сколько потребуется.
        :return:
        """
        log = log or mpfs.engine.process.get_default_log()
        self.max_processes = max_processes or multiprocessing.cpu_count()
        log.info('MAX_PROCESSES: %d' % self.max_processes)
        self._manager = multiprocessing.Manager()
        self._task_queue = self._manager.Queue(self.max_processes)
        self._target = target
        self._callback = callback
        self._wait_time = wait_time
        self._error_handler = error_handler
        self._termination_timeout = termination_timeout

        def worker_process(queue):
            obj = queue.get(True)
            while obj != TERMINAL_OBJECT:
                try:
                    log.info('Start %s on %s...' % (target.func_name, obj))
                    ret = target(obj)
                    if callback and hasattr(callback, '__call__'):
                        callback(ret)
                    log.info('%s on %s successfully completed.' % (target.func_name, obj))
                except Exception, e:
                    if error_handler and hasattr(error_handler, '__call__'):
                        error_handler(obj, e)
                    else:
                        try:
                            log.error(traceback.format_exc())
                            msg = 'Object: %s. Exception: %s' % (obj, e)
                            log.error(msg)
                        except Exception:
                            log.error("Cannot log exception type")

                queue.task_done()
                obj = queue.get(True)
            queue.task_done()

        self._workers = []
        for i in xrange(self.max_processes):
            worker = multiprocessing.Process(target=worker_process, args=(self._task_queue,))
            self._workers.append(worker)
            worker.start()

    def put(self, obj):
        wait = True
        while wait:
            try:
                self._task_queue.put(obj)
                wait = False
            except Queue.Full:
                time.sleep(self._wait_time)

    def terminate(self):
        # посылаем всем воркерам сигнал закругляться
        for w in self._workers:
            self.put(TERMINAL_OBJECT)

        if self._termination_timeout is not None:
            # если все процессы добровольно не закруглились, то ждём и даём им возможность корректно завершиться
            start = time.time()
            while [w for w in self._workers if w.is_alive()] and time.time() - start < self._termination_timeout:
                time.sleep(self._wait_time)

            # наше терпение лопнуло, начинаем звереть и убивать не успевших закруглиться по-хорошему воркеров
            for w in self._workers:
                if w.is_alive():
                    w.terminate()
                    # не забываем сообщать queue, что убитый воркер прекратил работать над задачей
                    self._task_queue.task_done()

        self._task_queue.join()


def get_uids_from_file(filename):
    with open(filename, 'rU+') as f:
        # отматываем курсор в конец файла
        f.seek(0, os.SEEK_END)
        uid = ''
        while f.tell() > 0:
            # шаг назад, чтоб считать новый символ
            f.seek(-1, os.SEEK_CUR)
            char = f.read(1)

            if char == '\n':
                if uid:
                    # если uid не пустой (например, пустая строка в конце файла),
                    # то пушим его в очередь на обработку и обнуляем
                    yield uid
                    uid = ''

                # шаг назад, чтоб отрубить при truncate разделитель uid'ов
                if f.tell() > 0:
                    f.seek(-1, os.SEEK_CUR)
                # удаляем строку со считанным uid'ом
                f.truncate(f.tell())
            else:
                # собираем uid посимвольно
                uid = '%s%s' % (char, uid)
                # шаг назад, чтоб скомпенсировать шаг вперёд выполненный read'ом
                f.seek(-1, os.SEEK_CUR)

        if uid:
            yield uid


def get_uids_from_db(log, **kwargs):
    user_type = kwargs.get('user_type', 'standart')
    collection = kwargs.get('collection', 'user_index')
    limit = kwargs.get('limit', 0)
    spec = kwargs.get('spec', {})

    db = mpfs.engine.process.dbctl().database()
    count = 0
    processed = 0

    user_count = db[collection].find(spec).count()
    if limit and limit < user_count:
        user_count = limit
    log.info('Found %d uids in %s. Start processing uids...' % (user_count, collection))

    # Тут нет постраничного выгребания, т.к. курсор не должен успевать умирать между запросами пачек.
    users = db[collection].find(spec, fields=('_id', 'type', 'uid')).limit(limit)
    for user in users:
        uid = user.get('uid') or user.get('_id')
        if uid:
            u_type = user.get('type', None)
            if user_type is None or u_type == user_type:
                count += 1
                yield uid
            else:
                log.info('Skip uid %s because user type %s != %s' % (uid, u_type, user_type))
            processed += 1

            # выводим в лог прогресс каждые 1000 записей
            if processed % 1000 == 0:
                log.info('Returned %d / %d uids...' % (count, user_count))

    log.info('Successfully processed %d / %d uids.' % (count, user_count))


def dump_db_uids_to_file(file_name, log, **kwargs):
    with open(file_name, 'w') as f:
        for uid in get_uids_from_db(log, **kwargs):
            f.write('%s\n' % uid)


def get_uids(file_name, log, **kwargs):
    if not os.path.exists(file_name):
        dump_db_uids_to_file(file_name, log, **kwargs)

    for uid in get_uids_from_file(file_name):
        yield uid


def run(process, callback, file_name, **kwargs):
    """
    Обработка функции process на всех пользователях типа user_type из file_name или из user_index.

    :param process:
    :param callback:
    :param file_name:
    :param max_processes:
    :param error_handler:
    :param user_type:
    :param log:
    :param termination_timeout:
    """
    max_processes = kwargs.get('max_processes')
    error_handler = kwargs.get('error_handler')
    log = kwargs.pop('log', None) or mpfs.engine.process.get_default_log()
    termination_timeout = kwargs.get('termination_timeout', None)

    uids = None

    try:
        tq = BaseTaskQueue(process, callback=callback, max_processes=max_processes, error_handler=error_handler,
                           log=log, termination_timeout=termination_timeout)

        for uid in get_uids(file_name, log, **kwargs):
            tq.put(uid)

        # завершаем все подпроцессы
        tq.terminate()
        # удаляем файл после успешной обработки всех uid'ов
        os.remove(file_name)
        log.info('File "%s" fully processed, removed' % file_name)
    except Exception, e:
        log.error('Error occurred while executing %s: %s' % (process, e))
        log.error(traceback.format_exc())
        log.error('Terminating.')
    finally:
        if uids is not None:
            uids.close()

