# -*- coding: utf-8 -*-
"""
Лимиты на разные показатели.
"""
from abc import ABCMeta, abstractmethod, abstractproperty

from mpfs.common.util import to_json, filter_uid_by_percentage
from mpfs.common.util.limits.errors import LimitExceeded


class AbstractLimit(object):
    __metaclass__ = ABCMeta

    NO_LIMIT = -1

    @abstractproperty
    def is_enabled(self):
        pass

    @abstractmethod
    def get_limit(self):
        pass


class AbstractValue(object):
    __metaclass__ = ABCMeta

    def is_exceeded(self, limit):
        current_limit = limit.get_limit()
        if current_limit == limit.NO_LIMIT:
            return False

        return self.get_value() > current_limit

    @abstractmethod
    def get_value(self):
        pass


class ConfigLimitMixin(object):
    config_path = None

    def __init__(self, uid, config):
        self.uid = uid
        self.config = config[self.config_path]

    @property
    def is_enabled(self):
        if not bool(self.config['enabled']):
            return False

        return filter_uid_by_percentage(self.uid, self.config['percentage'])


class AbstractLimitChecker(object):
    __metaclass__ = ABCMeta

    exception_cls = LimitExceeded
    reason = 'Limit exceeded'

    def __init__(self, limit, value):
        self.limit = limit
        self.value = value

    def limit_exceeded(self, value, limit):
        raise self.exception_cls(title=to_json({'reason': self.reason, 'limit': limit}))

    def check(self):
        if not self.limit.is_enabled:
            return

        if self.value.is_exceeded(self.limit):
            self.limit_exceeded(self.value.get_value(), self.limit.get_limit())
