# -*- coding: utf-8 -*-
import mpfs.engine.process

from mpfs.common.static.tags.experiment_names import UPLOAD_MAX_FILE_SIZE_LIMIT_ERRORS
from mpfs.common.util import to_json
from mpfs.common.util.experiments.logic import experiment_manager
from mpfs.common.util.limits.errors import UploadFileSizeLimitExceeded
from mpfs.common.util.limits.logic.base import AbstractValue, AbstractLimit, AbstractLimitChecker
from mpfs.core.filesystem.quota import Quota


class UploadFileSizeValue(AbstractValue):
    def __init__(self, size):
        self.size = size

    def get_value(self):
        return self.size


class LimitByPaid(AbstractLimit):
    def __init__(self, uid):
        self.uid = uid
        user_info = Quota().report(uid)
        self.file_size_limit = user_info['filesize_limit']
        self.paid_file_size_limit = user_info['paid_filesize_limit']

    @property
    def is_enabled(self):
        return True

    def get_limit(self):
        return self.file_size_limit


class UploadLimitChecker(AbstractLimitChecker):
    exception_cls = UploadFileSizeLimitExceeded
    reason = 'UploadFileSizeLimitExceeded'

    def limit_exceeded(self, value, limit):
        if value > self.limit.paid_file_size_limit:
            raise self.exception_cls(title=to_json({'reason': self.reason, 'limit': self.limit.paid_file_size_limit}))
        else:
            raise self.exception_cls(title=to_json({'reason': 'FreeUser' + self.reason, 'limit': limit}))


def check_upload_file_size_limits(uid, size):
    if not experiment_manager.is_feature_active(UPLOAD_MAX_FILE_SIZE_LIMIT_ERRORS):
        return

    if not mpfs.engine.process.usrctl().is_user_init(uid):
        return

    from mpfs.core.user.base import User
    from mpfs.core.user.attach import AttachUser
    user = User(uid)
    if isinstance(user, AttachUser):
        return

    checker = UploadLimitChecker(limit=LimitByPaid(uid), value=UploadFileSizeValue(size))
    checker.check()
