# -*- coding: utf-8 -*-
import mpfs.engine.process

from mpfs.common.static.tags.experiment_names import UPLOAD_TRAFFIC_LIMIT_ERRORS
from mpfs.common.util import to_json
from mpfs.common.util.datetime_util import SafeTimestampDateTime
from mpfs.common.util.experiments.logic import experiment_manager
from mpfs.common.util.limits.errors import UploadTrafficLimitExceeded, UploadTrafficLimitWithoutDetailsError
from mpfs.common.util.limits.logic.base import AbstractLimit, ConfigLimitMixin, AbstractValue, AbstractLimitChecker
from mpfs.common.util.limits.utils import UploadLimitPeriods
from mpfs.common.util.size.datasize import DataSize
from mpfs.config import settings
from mpfs.core.filesystem.quota import Quota
from mpfs.core.user.base import User

class UploadTrafficValue(AbstractValue):
    def __init__(self, uid, period, traffic_diff):
        self.uid = uid
        self.period = period

        self.diff = 0
        if traffic_diff:
            self.diff = long(traffic_diff)

    def get_value(self):
        return Quota().upload_traffic(self.uid, self.period) + self.diff


class LimitByQuota(ConfigLimitMixin, AbstractLimit):
    config_path = 'by_quota'

    def __init__(self, uid, config):
        super(LimitByQuota, self).__init__(uid=uid, config=config)
        user_info = Quota().report(uid)
        self.limit = user_info['limit'] * float(self.config['multiplier'])

    def get_limit(self):
        return self.limit


class LimitByPaid(ConfigLimitMixin, AbstractLimit):
    config_path = 'by_paid'

    def get_limit(self):
        user = User(self.uid)
        if user.is_paid():
            return DataSize.parse(self.config['limits']['paid']).to_bytes()
        else:
            return DataSize.parse(self.config['limits']['free']).to_bytes()


class LimitByUID(ConfigLimitMixin, AbstractLimit):
    config_path = 'by_uid'

    def get_limit(self):
        if self.uid in self.config['limits']:
            return DataSize.parse(self.config['limits'][self.uid]).to_bytes()
        else:
            return self.NO_LIMIT


class LimitByRegTime(ConfigLimitMixin, AbstractLimit):
    config_path = 'by_reg_time'

    def get_limit(self):
        user = User(self.uid)
        reg_dt = SafeTimestampDateTime.fromtimestamp(user.reg_time)
        if str(reg_dt) > str(self.config['reg_time']):
            return DataSize.parse(self.config['limit']).to_bytes()
        else:
            return self.NO_LIMIT


class UploadLimitChecker(AbstractLimitChecker):
    exception_cls = UploadTrafficLimitExceeded
    exception_without_details_cls = UploadTrafficLimitWithoutDetailsError

    @property
    def reason(self):
        return '%sUploadTrafficLimitExceeded' % self.value.period.name.capitalize()

    def limit_exceeded(self, value, limit):
        if self.limit.config['error_with_details']:
            super(UploadLimitChecker, self).limit_exceeded(value=value, limit=limit)
        else:
            raise self.exception_without_details_cls(title=to_json({'reason': 'UserReadOnly'}))


def check_upload_limits(uid, size):
    if not experiment_manager.is_feature_active(UPLOAD_TRAFFIC_LIMIT_ERRORS):
        return

    if not mpfs.engine.process.usrctl().is_user_init(uid):
        return

    from mpfs.core.user.base import User
    from mpfs.core.user.attach import AttachUser
    user = User(uid)
    if isinstance(user, AttachUser):
        return

    for period in UploadLimitPeriods:
        if (period.value.name not in settings.limits['upload_traffic'] or
                not settings.limits['upload_traffic'][period.value.name]['counter_enabled']):
            continue
        value = UploadTrafficValue(uid=uid, period=period.value, traffic_diff=size)
        for limit_cls in [LimitByQuota, LimitByUID, LimitByRegTime, LimitByPaid]:
            limit = limit_cls(uid, settings.limits['upload_traffic'][period.value.name])
            checker = UploadLimitChecker(limit=limit, value=value)
            checker.check()
