# -*- coding: utf-8 -*-

"""
Утилиты для логирования в МПФС.

Все логгеры в МПФС организуются в иерархическую структуру. Базовым для МПФС логгеров является логгер с именем `mpfs`.
Он является корнем поддерева и пространством имен одновременно.

Структура логгеров в МПФС:
              root
               |-- mpfs
               |     -- core.*
               |    |   common.*
     уровень   |    |   engine.*
     модулей   |    |   frontend.*
               |    |    . . .
               |     -- platform.*
               |     -- error
               |    |   access
     спец.     |    |   service
     наз.      |    |   requests
               |     -- stat
               |-- event-history
               |-- stat-store
     устаревшие|    .  .  .
               |-- storage-cleaner-worker

Обращение к логгерам производится по имени: "mpfs", "mpfs.core" и т.д.

Все логгеры создатся атвоматически при вызове функции :func:`configure_logging` или
при первом обращении к ним по имени. Для удобства определены функции в модуле :mod:`mpfs.engine.process`:
    * get_default_log  - возвращает логгер уровня модулей
    * get_error_log    - возвращает логгер необработанных ошибок
    * get_requests_log - возвращает логгер запросов в СУБД и сторонние сервисы
    * get_access_log   - возвращает логгер входных запросов
    * get_service_log  - возвращает логгер ответов запросов в сторонние сервисы
    * get_stat_log     - возвращает логгер статистики. Формат без схемы.

"""

import inspect
import logging
import logging.config
import os
import signal
import threading
import time

from functools import partial

import mpfs.engine.process

_monitor = None


class MPFSLogger(logging.getLoggerClass()):
    """Класс всех логгеров МПФС (кроме root).

    При создании каждой записи добавляет в нее специфичные для МПФС атрибуты:
    * mpfs_appname - имя приложения, которое исопльзует кодовую базу МПФС.
    * mpfs_request_id - идентификатор запроса внутри МПФС или идентификатор задачи в очередях.
    * mpfs_ycrid - идентификатор запроса внутри Яндекса.

    Таким образом, любая запись обладает информацией в рамках какого приложения,
    запроса внутри приложения и запроса внутри Яндекса она была создана.
    """

    # Celery, уходи.
    # https://github.com/celery/celery/blob/3.1/celery/utils/log.py#L271-272
    _signal_safe = True
    _process_aware = True

    def makeRecord(self, name, level, fn, lno, msg, args, exc_info, func=None, extra=None):

        get_req_id = mpfs.engine.process.get_req_id
        get_cloud_req_id = mpfs.engine.process.get_cloud_req_id
        get_app_name = mpfs.engine.process.get_app_name
        get_host_name = mpfs.engine.process.hostname

        new_extra = {}
        # `mpfs_*`, чтобы отделить от служебных полей.
        mpfs_extra = {
            'mpfs_request_id': get_req_id() or '',
            'mpfs_ycrid': get_cloud_req_id() or '',
            'mpfs_appname': get_app_name() or '',
            'mpfs_host_name': get_host_name() or ''
        }

        if extra is not None:
            keys = mpfs_extra.viewkeys() & extra.viewkeys()
            if keys:
                raise KeyError("Attempt to overwrite %r in `mpfs_extra`" % keys)
            new_extra.update(extra)
        new_extra.update(mpfs_extra)
        return super(MPFSLogger, self).makeRecord(name, level, fn, lno, msg, args, exc_info, func, extra=new_extra)


class TSKVSafeFormatter(logging.Formatter):
    """Форматтер, учитывающий спецификацию TSKV формата.

    Делает безопасным форматирование записей в TSKV,
    с учетом трейсов ошибок и стека.
    """

    def format(self, record):
        """Форматировать запись, безопасно для TSKV.

        :type record: :class:`logging.Record`
        :rtype: str | unicode
        """
        # TODO заюзать statbox_bindings.tskv ? Оно сильно быстрее должно быть
        tskv_esc = TSKVMessage.escape_special
        decode = partial(unicode, encoding='utf-8', errors='strict')

        record.message = record.getMessage()
        if self.usesTime():
            record.asctime = self.formatTime(record, self.datefmt)
        try:
            s = self._fmt % tskv_esc(record.__dict__)
        except UnicodeError:
            try:
                s = self._fmt % tskv_esc(record.__dict__, decode)
            except UnicodeDecodeError:
                raise

        if record.exc_info:
            # Кэшируем трейс, т.к. он сохраняется между логгерами.
            if not record.exc_text:
                record.exc_text = self.formatException(record.exc_info)
        if record.exc_text:
            if s[-1:] != r"\n":
                s += r"\n"
            try:
                s = s + tskv_esc(record.exc_text)
            except UnicodeError:
                try:
                    s = s + tskv_esc(decode(record.exc_text, errors='replace'))
                except UnicodeDecodeError:
                    raise
        return s


class TSKVSafeLimitedFormatter(TSKVSafeFormatter):
    """TSKV форматтер с ограничением по максимальной длине строки лога.

    """
    TAIL_OFFSET = 1000

    def __init__(self, max_message_length=10**12, **kwargs):
        super(TSKVSafeLimitedFormatter, self).__init__(**kwargs)
        self.max_message_length = max_message_length

    def format(self, record):
        if len(record.msg) > self.max_message_length:
            record.msg = self._truncate(record.msg)
        return super(TSKVSafeLimitedFormatter, self).format(record)

    def _truncate(self, message):
        return '%s...<TRUNCATED %d>...%s' % (
            message[:self.max_message_length - self.TAIL_OFFSET],
            len(message) - self.max_message_length,
            message[-self.TAIL_OFFSET:]
        )


class StatLoggerAdapter(logging.LoggerAdapter):
    """Адаптер для логгера статистики.

    Облегачает логирование в TSKV для произвольного формата:
        * игнорирует `exc_info`, `extra`
        * конвертирует `msg`(dict) в `TSKVMessage`
        * добавляет обязательные для TSKV поля

    .. doctest::
        >>> stat_log = StatLoggerAdapter(logging.getLogger())
        >>> stat_log.info({'a': 1, 'b': 'hello','c': '12\tfoo'})
        tskv	tskv_format=ydisk-mpfs-stat-log	unixtime=1464009941	a=1    b=Hello    c=12\tfoo

        >>> video_stat_log = StatLoggerAdapter(logging.getLogger(), postfix='video')
        >>> video_stat_log.info({'a': 1, 'b': 'hello','c': '12\tfoo'})
        tskv	tskv_format=ydisk-mpfs-stat-log-video	unixtime=1464009945	a=1	c=12\tfoo	b=hello
    """

    def __init__(self, logger, name=''):
        """Инициализировать адаптер для логгера статистики.

        Значение `name` устанавливается как постфикс для `tskv_format` в записи.
        Т.е.: ydisk-mpfs-stat-log-{постфикс}.

        С помощью этого значения отделяются отчеты внутри таблицы ydisk-mpfs-stat-log.

        :type logger: :class:`logging.Logger`
        :type name: str
        """
        if name and not name.startswith('-'):
            name = '-' + name
        super(StatLoggerAdapter, self).__init__(logger, {'postfix': name})

    def process(self, msg, kwargs):
        if not isinstance(msg, dict):
            raise TypeError('`dict` is required, got `%s` instead' % type(msg))

        reserved = {'tskv', 'tskv_format', 'unixtime', 'postfix', 'ident'}
        if reserved & msg.viewkeys():
            raise ValueError('Attempt to overwrite reserved fields: %r' % (reserved & msg.viewkeys()))

        msg = TSKVMessage.with_special_escaped(**msg)
        return super(StatLoggerAdapter, self).process(msg, {})


def rotate_logs(signum, frame):
    """Закрыть дескрипторы обработчиков логов.

    :type signum: int
    :type frame: :class:`types.FrameType`
    """
    print '[%d][rotate_logs] Got signal `%d`, closing handlers..' % (os.getpid(), signum)
    try:
        close_log_handlers()
    except Exception as e:
        print '[%d][rotate_logs] Rotation failed: %s' % (os.getpid(), e)
        return
    print '[%d][rotate_logs] Rotation is done' % os.getpid()


def close_log_handlers():
    """Закрыть обработчики логов.

    Рассчитываем, что хендлер способен переоткрыться при записи.
    Для этого используем :class:`logging.handlers.WatchedFileHandler`
    """

    for name, logger in logging.Logger.manager.loggerDict.iteritems():
        if not isinstance(logger, logging.PlaceHolder) and logger.handlers:
            for h in logger.handlers:
                try:
                    h.close()
                except ValueError as e:
                    print '[%d][close_log_handlers] Logger `%s` got error: %s' % (os.getpid(), name, e)


def configure_logging(dict_config, app_name=None, signum=signal.SIGRTMIN):
    """Настроить подсистему логирования МПФС.

    Вызывается единственный раз в точке входа в приложение, до большинства импортов.
        * устанавливает класс логгера.
        * настраивает `handlers`, `formatters`, `filters`, `loggers` через
            стандартную библиотеку :mod:`logging`.
        * устанавливает ротацию по сигналу `signum`
        * устанавливает глобальное имя приложения `app_name`.

    Это рекомендуемый способ настраивать логирование для приложений, использующих
    кодовую базу МПФС.

    Полезные ссылки:
        `logging.dictConfig`
        https://docs.python.org/2/library/logging.config.html#dictionary-schema-details

    :type dict_config: dict
    :type app_name: None | str
    :type signum: None | int
    """

    if not isinstance(dict_config, dict):
        raise TypeError("`dict` is required, got `%s` instead" % type(dict_config))

    logging.setLoggerClass(MPFSLogger)
    logging.config.dictConfig(dict_config)

    if signum is not None:
        signal.signal(signum, rotate_logs)
        signal.siginterrupt(signum, False)  # https://st.yandex-team.ru/CHEMODAN-28340

    if app_name is not None:
        mpfs.engine.process.set_app_name(app_name)


def enable_monitor():
    # только для того чтобы не протухал weakref
    global _monitor
    _monitor = Monitor()

    if mpfs.engine.process.is_uwsgi_process():
        register_after_fork = mpfs.engine.process.get_register_after_fork_impl()
        register_after_fork(_monitor, Monitor.start)
    else:
        _monitor.start()


class Monitor(threading.Thread):
    def __init__(self, period=300):
        super(Monitor, self).__init__()
        self._period = float(period)
        self._total_close_count = 0
        self._log_close_status_frequency = 5
        self.setDaemon(True)

    def run(self):
        while True:
            self._sleep()
            self._close_handlers()

    def _sleep(self):
        try:
            time.sleep(self._period)
        except Exception as e:
            mpfs.engine.process.get_error_log().error(
                "[logger.Monitor] Gor error while sleep: %s", e)

    def _close_handlers(self):
        try:
            self._total_close_count += 1
            close_log_handlers()
            if self._total_close_count % self._log_close_status_frequency == 0:
                mpfs.engine.process.get_default_log().info(
                    "[logger.Monitor] Handlers were closed %d times." % self._total_close_count)
        except Exception as e:
            mpfs.engine.process.get_error_log().error(
                "[logger.Monitor] Got error on closing handlers: %s", e)


def redirect_logs(get_handler):
    """Заменить всем логгерам обработчик логов с сохранением уровня и форматтера.

    Используется для переопределения файла, в который пишется лог.
    Удобно в тестах, чтобы направить все логи в один файл.

    :type get_handler: :class:`collections.Callable`
    """
    for name, logger in logging.Logger.manager.loggerDict.iteritems():
        if not isinstance(logger, logging.PlaceHolder):
            for handler in logger.handlers:
                new_handler = get_handler()
                new_handler.setLevel(handler.level)
                new_handler.setFormatter(handler.formatter)
                handler.close()
                logger.removeHandler(handler)
                logger.addHandler(new_handler)


def merge_dict_configs(main, concrete):
    """Обновить элементы `main` элементами из `concrete`.

    Учитывая специфику словаря `dictConfig`, обновление происходит на
    одном уровне вложенности.

    :type main: dict
    :type concrete: dict
    :rtype: dict
    """
    merged = {}
    merged.update(main)
    for k in concrete:
        if isinstance(concrete[k], dict):
            merged[k].update(concrete[k])
        else:
            merged[k] = concrete[k]
    return merged


def get(name):
    """Получить объект класса логгера с именем `name`.

    Если `name` is ``None``, то вернется объект :class:`logging.RootLogger`

    :type name: None | str
    :rtype: :class:`logging.RootLogger` | :class:`~MPFSLogger`
    """
    return logging.getLogger(name)


def log_time(_log):
    def wrap(f):
        def wrapped_f(*args, **kwargs):
            start = time.time()
            e = None
            try:
                result = f(*args, **kwargs)
            except Exception, e:
                pass
            end = time.time() - start
            _log.info('TIME_CHECK [time: %.5f sec] [method: %s] [called from: %s]' % \
                      (end, f.func_name,
                       'module: %s line: %s parent_method: %s' % inspect.getouterframes(inspect.currentframe(), 2)[1][1:4]
                       ))
            if e:
                raise e
            else:
                return result
        return wrapped_f
    return wrap


# stat-store is log with special message format used in number of places in project
# so define its template and interface here to avoid duplication in code.

STAT_LOG_TEMPLATE = ('U:%(uid)s T:%(operation_type)s D:%(operation_subtype)s HL:%(hardlinked)s S:%(size)s '
                     'MT:%(media_type)s Y:%(ycrid)s TIME:%(log_time)s P:%(path)s')


def log_stat(uid, op_type, op_subtype, hardlinked, res_size, res_media_type, path=''):
    ycrid = mpfs.engine.process.get_cloud_req_id() or ''
    log_time = str(int(time.time()))
    log_data = {'uid': uid, 'operation_type': op_type, 'operation_subtype': op_subtype, 'hardlinked': hardlinked,
                'size': res_size, 'media_type': res_media_type, 'ycrid': ycrid, 'log_time': log_time, 'path': path}
    get("stat-store").info(STAT_LOG_TEMPLATE, log_data)
    mpfs.engine.process.get_stat_log('store').info(log_data)


def suppress_log(log):
    """
    :param log: Логгер который следует заткнуть.
    """
    def decorator(fn):
        def wrapper(*args, **kwargs):
            stub = lambda *args, **kwargs: None
            attrs_names = ('info', 'debug', 'warning', 'error', 'critical', 'log', 'exception')
            old_attrs = {}
            for attr in attrs_names:
                old_attrs[attr] = getattr(log, attr, None)
                setattr(log, attr, stub)
            try:
                ret = fn(*args, **kwargs)
            finally:
                # Возвращаем всё как было
                for attr in attrs_names:
                    setattr(log, attr, old_attrs[attr])
            return ret
        return wrapper
    return decorator


def suppress_mongo_requests_log(fn):
    """Декоратор вырубающий запись монговских запросов в mpfs.requests логгер"""
    return suppress_log(mpfs.engine.process.get_requests_log())(fn)


def suppress_error_log(fn):
    """Декоратор вырубающий запись в mpfs.error логгер"""
    return suppress_log(mpfs.engine.process.get_error_log())(fn)


def suppress_default_log(fn):
    """Декоратор вырубающий запись в mpfs.default логгер"""
    return suppress_log(mpfs.engine.process.get_default_log())(fn)


class TSKVMessage(object):
    '''
    Преобразует переданные в конструкторе параметры в tskv запись

    Usage:
    print TSKVMessage('a', b='c', d='r\td')
    arg_0=a b=c     d=r\td
    '''
    _kv_delimiter = u'='
    _delimiter = u'\t'
    _args_key_template = u'arg_%i'
    _special_substitutions = [
        ('\\', '\\\\'),
        ('\t', r'\t'),
        ('\r', r'\r'),
        ('\n', r'\n'),
        ('\0', r'\0'),
    ]

    def __init__(self, *args, **kwargs):
        self.args = args
        self.kwargs = kwargs

    @classmethod
    def with_special_escaped(cls, *args, **kwargs):
        return cls(*cls.escape_special(args), **cls.escape_special(kwargs))

    def update(self, *args, **kwargs):
        self.args += args
        self.kwargs.update(kwargs)
        return self

    @staticmethod
    def quote_value(value):
        if isinstance(value, unicode):
            return value
        elif isinstance(value, str):
            return value.decode('utf-8')
        else:
            return unicode(value)

    @staticmethod
    def quote_key(key):
        return TSKVMessage.quote_value(key).replace(u'=', u'\=')

    @classmethod
    def build_kv_pair(cls, key, value):
        return u"%s%s%s" % (key, cls._kv_delimiter, value)

    # https://wiki.yandex-team.ru/statbox/LogRequirements
    @classmethod
    def escape_special(cls, data, decode=None):
        """В строках, внутри `data` экранировать небезопасные с т.зр. TSKV символы.

         Если строка имеет тип :class:`str` и указан `decode`, то
         сделать преобразование в unicode по правилам `decode`.

        .. doctest::
            >>> from functools import partial
            >>> decode = partial(unicode, encoding='utf-8', errors='strict')
            >>> TSKVMessage.escape_special('Привет', decode)
            u'\u041f\u0440\u0438\u0432\u0435\u0442'
            >>> TSKVMessage.escape_special({'a': 'Привет'})
            {'a': '\xd0\x9f\xd1\x80\xd0\xb8\xd0\xb2\xd0\xb5\xd1\x82'}

        :type data: Any
        :type decode: :class:`collections.Callable` | None
        :rtype: Any
        """
        if isinstance(data, basestring):
            for search, replacement in cls._special_substitutions:
                data = data.replace(search, replacement)
            if isinstance(data, str) and decode:
                data = decode(data)
        elif isinstance(data, dict):
            new_data = {}
            for key, value in data.iteritems():
                new_data[key] = cls.escape_special(value, decode)
            data = new_data
        elif isinstance(data, (list, tuple)):
            new_data = []
            for value in data:
                new_data.append(cls.escape_special(value, decode))
            if isinstance(data, tuple):
                data = tuple(new_data)
            else:
                data = new_data
        return data

    def __unicode__(self):
        result = []
        for i, value in enumerate(self.args):
            key = self.quote_key(self._args_key_template % i)
            value = self.quote_value(value)
            result.append(self.build_kv_pair(key, value))
        for key, value in self.kwargs.iteritems():
            key = self.quote_key(key)
            value = self.quote_value(value)
            result.append(self.build_kv_pair(key, value))
        return self._delimiter.join(result)

    def __str__(self):
        return unicode(self).encode('utf-8')

    @classmethod
    def get_delimiter(cls):
        return cls._delimiter


def headers_to_log(headers, not_logged_headers=None):
    from mpfs.common.util.credential_sanitizer import CredentialSanitizer

    headers = dict(headers)
    if not_logged_headers:
        headers = {k: v
                   for k, v in headers.iteritems()
                   if k.lower() not in not_logged_headers}
    hashed_headers_list = CredentialSanitizer.get_headers_list_with_sanitized_credentials(headers)
    return str(dict(hashed_headers_list))
