# -*- coding: utf-8 -*-
"""

MPFS
UTIL

SMTP-агент

"""
import smtplib
import os
import re
import traceback
import time
import hashlib
import base64

from itertools import imap
from email.MIMEMultipart import MIMEMultipart
from email.MIMEText import MIMEText
from email.MIMEImage import MIMEImage
from email.header import Header
from email.utils import formatdate

import mpfs.engine.process

from mpfs.config import settings
from mpfs.common.util.templater import TemplateProcessor

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

DEFAULT_LOCALE = settings.user['default_locale']
TEMPLATE_DIR = settings.system['fs_paths']['template_dir']
STATIC_DIR = 'base/'
PLAIN_TEMPLATE = 'plain_template'


def _read_file(directory, name):
    """
    Прочесть произвольный файл
    """
    fp = open(os.path.join(TEMPLATE_DIR, directory, name), 'rb')
    contents = fp.read()
    fp.close()
    return contents


def _read_text_file(template, name):
    """
    Прочесть текстовый файл
    """
    text = _read_file(template, name)
    return text.decode('utf-8')


def get_yandex_service_header(hdate, hfrom, hsubj):
    """
    Выдать яндексовый заголовок
    """
    first_string = '%s_%s_%s_%s_%s' % (
    hdate, hfrom.encode('utf-8'), hsubj, settings.X_YANDEX_LABEL, settings.X_YANDEX_SALT)
    md5 = hashlib.md5()
    md5.update(first_string)
    second_string = '%s %s' % (settings.X_YANDEX_LABEL, md5.hexdigest())
    return base64.b64encode(second_string)


def template_args(template):
    src_text = _read_text_file(template, 'html.' + DEFAULT_LOCALE)
    base_text = _read_text_file('base', 'base') + \
                _read_text_file('base', 'base-html.' + DEFAULT_LOCALE) + \
                _read_text_file('base', 'base-plain.' + DEFAULT_LOCALE)

    base_args = re.compile('{%- set ([\w\d\-\.]+)').findall(base_text)
    src_args = re.compile('{{([\w\d\-\.]+)}}').findall(src_text)

    return filter(lambda x: x not in base_args, src_args)


def sendmail(smtp, from_addr, to_addrs, msg, mail_options=[], rcpt_options=[]):
    # the copy of smtp.sendmail method but returns status and responses of all sends, not only failed
    smtp.ehlo_or_helo_if_needed()
    esmtp_opts = []
    if smtp.does_esmtp:
        # Hmmm? what's this? -ddm
        # self.esmtp_features['7bit']=""
        if smtp.has_extn('size'):
            esmtp_opts.append("size=%d" % len(msg))
        for option in mail_options:
            esmtp_opts.append(option)
    res = smtp.mail(from_addr, esmtp_opts)
    (code, resp) = res
    log.info('smtp method mail, code: %s resp: %s' % (code, resp))  # change 0
    if code != 250:
        smtp.rset()
        raise smtplib.SMTPSenderRefused(code, resp, from_addr)
    responses = {} #change 1
    senderrs = {}
    if isinstance(to_addrs, basestring):
        to_addrs = [to_addrs]
    for each in to_addrs:
        (code, resp) = smtp.rcpt(each, rcpt_options)
        responses[each] = (code, resp) #change 2
        if (code != 250) and (code != 251):
            senderrs[each] = (code, resp)
    if len(senderrs) == len(to_addrs):
        # the server refused all our recipients
        smtp.rset()
        raise smtplib.SMTPRecipientsRefused(senderrs)
    (code, resp) = smtp.data(msg)
    log.info('smtp method data, code: %s resp: %s' % (code, resp))  # change 3
    if code != 250:
        smtp.rset()
        raise smtplib.SMTPDataError(code, resp)
    # if we got here then somebody got our mail
    return responses #change 4


def send(to, template, args=None, sender_email=None, sender_name=None, headers=None):
    """
        Для отправки письма без без какого-либо форматирования нужно:
            1. Указать 'plain_template' в качестве параметра template;
            2. Передать в args['subject'] заголовок письма;
            3. Передать в args['body'] тело письма;
    """
    if not args:
        args = {}
    if 'currentYear' not in args:
        args['currentYear'] = str(time.localtime(time.time())[0])

    # защита от падений в случаях, когда паспорт вернул пустой email(https://st.yandex-team.ru/CHEMODAN-24272)
    if not isinstance(to, basestring) or '@' not in to:
        error_log.error("Error: invalid email: %s" % to)
        return

    log.debug('template args: %s' % args)

    # load templates
    if template == PLAIN_TEMPLATE:
        subject = args['subject'].encode('utf-8')
        plainText = args['body'].encode('utf-8')
    else:
        templater = TemplateProcessor(template, args)
        subject = templater.process('subject')
        htmlText = templater.process('html')
        plainText = templater.process('plain')

    # process from, to
    if not sender_email:
        if templater.locale == 'ru':
            sender_email = settings.smtp['from_ru']
        else:
            sender_email = settings.smtp['from_com']

    if not sender_name:
        sender_name = templater.process('from')
    elif isinstance(sender_name, unicode):
        sender_name = sender_name.encode('utf-8')

    # Create the root message and fill in the from, to, and subject headers
    sendtime = formatdate(time.time(), True)

    msgRoot = MIMEMultipart('related')
    if headers:
        for key, value in headers.iteritems():
            msgRoot[key] = value

    msgRoot['Subject'] = Header(subject.decode('utf-8'), 'utf-8')
    msgRoot['From'] = '%s <%s>' % (str(Header(sender_name.decode('utf-8'), 'utf-8')), sender_email)
    msgRoot['To'] = to
    msgRoot['Date'] = sendtime
    msgRoot['X-Yandex-Service'] = get_yandex_service_header(sendtime, msgRoot['From'], subject)
    msgRoot['List-ID'] = template

    # CC field
    if 'cc' in args and len(args['cc']):
        msgRoot['Cc'] = ','.join(args['cc'])

    if template == PLAIN_TEMPLATE:
        msgText = MIMEText(plainText)
        msgText.set_charset('utf-8')
        msgRoot.attach(msgText)
    else:
        # preamble
        msgRoot.preamble = 'This is a multi-part message in MIME format.'

        # Encapsulate the plain and HTML versions of the message body in an
        # 'alternative' part, so message agents can decide which they want to display.
        msgAlternative = MIMEMultipart('alternative')
        msgRoot.attach(msgAlternative)

        msgText = MIMEText(plainText)
        msgText.set_charset('utf-8')
        msgAlternative.attach(msgText)

        # We reference the image in the IMG SRC attribute by the ID we give it below
        msgText = MIMEText(htmlText, 'html')
        msgText.set_charset('utf-8')
        msgAlternative.attach(msgText)

        # Get image cids
        required_cids = re.compile('src=\"cid\:([\w\-]+)\"').findall(htmlText)

        # Add images
        for filename in os.listdir(os.path.join(TEMPLATE_DIR, STATIC_DIR)):
            chunks = filename.split('.')
            if chunks[-1] not in ('jpg', 'png', 'gif'): continue
            imgId = chunks[0]

            if imgId in required_cids:
                imgData = _read_file(os.path.join(TEMPLATE_DIR, STATIC_DIR), filename)
                msgImage = MIMEImage(imgData)
                msgImage.add_header('Content-ID', '<' + imgId + '>')
                msgRoot.attach(msgImage)

    # IDNA magic!
    name, domain = to.split('@')
    to = '.'.join(imap(lambda x: x.encode('idna'), name.split('.'))) + '@' + domain.encode('idna')

    # log.info(msgRoot.as_string())

    exception = None
    for try_count in (1, 2):
        try:
            smtp = smtplib.SMTP()
            smtp.connect(settings.smtp['host'])
            results = sendmail(smtp, sender_email, to, msgRoot.as_string())
            for to_address, (code, resp) in results.items():
                log.info('email to %s processed, template %s, smtp code %s, response %s' %
                         (to_address, template, code, resp))
            smtp.quit()
            return msgRoot
        except smtplib.SMTPRecipientsRefused as e:
            log.info('email %s from %s to %s was not sent: All recipient addresses refused. SMTPRecipientsRefused %s' %
                     (template, sender_email, to, e.recipients))
            return msgRoot
        except smtplib.SMTPDataError as e:
            if 'Message rejected under suspicion of SPAM' in e.smtp_error:
                error_log.info('email %s from %s to %s was not sent, code %s error: %s' % (template, sender_email, to,
                                                                            e.smtp_code, e.smtp_error))
                error_log.info(traceback.format_exc())
                return msgRoot
            else:
                error_log.info('sending email to %s failed with code %s, error %s try: %s' %
                               (to, e.smtp_code, e.smtp_error, try_count))
                error_log.info(traceback.format_exc())
                exception = e
        except Exception, e:
            error_log.info('sending email to %s failed, try: %s' % (to, try_count))
            error_log.info(traceback.format_exc())
            exception = e

    raise exception
